package ru.yandex.travel.gen_logfeller_parser_pojo.generator;

import java.util.ArrayList;
import java.util.Deque;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.BeanDescription;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.introspect.AnnotatedMember;
import com.fasterxml.jackson.databind.introspect.BeanPropertyDefinition;

import ru.yandex.travel.gen_logfeller_parser_pojo.annotations.LogfellerConfig;
import ru.yandex.travel.gen_logfeller_parser_pojo.annotations.LogfellerDecompose;
import ru.yandex.travel.gen_logfeller_parser_pojo.annotations.LogfellerIgnore;
import ru.yandex.travel.gen_logfeller_parser_pojo.generator.model.FieldDescription;
import ru.yandex.travel.gen_logfeller_parser_pojo.generator.model.JsonParserConfig;
import ru.yandex.travel.gen_logfeller_parser_pojo.model.FieldDemand;
import ru.yandex.travel.gen_logfeller_parser_pojo.model.FieldType;


public class JsonParserConfigBuilder {

    private final static ObjectMapper objectMapper = new ObjectMapper();

    static {
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.enable(SerializationFeature.INDENT_OUTPUT);
    }

    public static String toJsonString(Class<?> clazz) throws JsonProcessingException {
        return objectMapper.writeValueAsString(buildConfig(clazz));
    }

    public static JsonParserConfig buildConfig(Class<?> clazz) {
        final List<FieldDescription> descriptions = new ArrayList<>();

        buildConfigRecursive(clazz, new LinkedList<>(), new LinkedList<>(), descriptions);

        return new JsonParserConfig(descriptions);
    }

    private static void buildConfigRecursive(Class<?> clazz, Deque<String> nameStack, Deque<String> pathStack, List<FieldDescription> descriptions) {

        final JavaType javaType = objectMapper.getTypeFactory().constructType(clazz);
        final BeanDescription description = objectMapper.getSerializationConfig().introspect(javaType);
        final List<BeanPropertyDefinition> properties = description.findProperties();

        for (BeanPropertyDefinition property : properties) {
            final AnnotatedMember field = Objects.nonNull(property.getField()) ? property.getField() : property.getGetter();

            if (Objects.isNull(field)) {
                throw new IllegalStateException(String.format("Cannot get field or getter for property %s of class %s", property, clazz.getCanonicalName()));
            }

            final LogfellerIgnore logfellerIgnore = field.getAnnotation(LogfellerIgnore.class);
            if (Objects.nonNull(logfellerIgnore)) {
                continue;
            }

            String name;
            String path;
            FieldType type;
            FieldDemand demand;

            final LogfellerConfig config = field.getAnnotation(LogfellerConfig.class);
            if (Objects.nonNull(config) && !config.name().isEmpty()) {
                path = field.getName();
                name = config.name();
            } else {
                path = null;
                name = field.getName();
            }

            if (Objects.nonNull(config) && !config.type().equals(FieldType.AUTO)) {
                type = config.type();
            } else {
                type = getType(field.getRawType());
            }

            if (Objects.nonNull(config) && !config.demand().equals(FieldDemand.DEFAULT)) {
                demand = config.demand();
            } else {
                demand = null;
            }

            nameStack.addLast(name);
            if (Objects.isNull(path)) {
                pathStack.addLast(name);
            } else {
                pathStack.addLast(path);
            }

            final LogfellerDecompose decompose = field.getAnnotation(LogfellerDecompose.class);
            if (Objects.nonNull(decompose)) {
                if (!type.equals(FieldType.VT_ANY)) {
                    throw new IllegalStateException(String.format("Cannot decompose %s type", type));
                }
                buildConfigRecursive(field.getRawType(), nameStack, pathStack, descriptions);
            } else {
                final String fullName = String.join("_", nameStack);
                final String fullPath = pathStack.size() == 1 && Objects.isNull(path) ? null : String.join("/", pathStack);

                descriptions.add(new FieldDescription(
                        fullName,
                        fullPath,
                        type,
                        demand
                ));
            }

            pathStack.removeLast();
            nameStack.removeLast();
        }
    }

    private static <T> FieldType getType(Class<T> fieldClass) {
        if (fieldClass.equals(String.class)) {
            return FieldType.VT_STRING;
        }
        else if (fieldClass.equals(Integer.class) || fieldClass.equals(int.class)) {
            return FieldType.VT_INT32;
        }
        else if (fieldClass.equals(Long.class) || fieldClass.equals(long.class)) {
            return FieldType.VT_INT64;
        }
        else if (fieldClass.equals(Float.class) || fieldClass.equals(float.class) || fieldClass.equals(Double.class) || fieldClass.equals(double.class)) {
            return FieldType.VT_DOUBLE;
        }
        else if (fieldClass.equals(Boolean.class) || fieldClass.equals(boolean.class)) {
            return FieldType.VT_BOOLEAN;
        }
        else {
            return FieldType.VT_ANY;
        }
    }
}
