package main

import (
	"context"
	"log"
	"math/rand"
	"time"

	"github.com/yandex/pandora/cli"
	"github.com/yandex/pandora/core"
	"github.com/yandex/pandora/core/aggregator/netsample"
	coreimport "github.com/yandex/pandora/core/import"
	"github.com/yandex/pandora/core/register"
	"google.golang.org/grpc"

	"a.yandex-team.ru/travel/hotels/proto/geocounter_service"
	pb "a.yandex-team.ru/travel/hotels/proto2"
)

type Ammo struct {
	Tag string

	LowerLeftLat  float64
	LowerLeftLon  float64
	UpperRightLat float64
	UpperRightLon float64

	FixedSize          float64
	BigFixedSize       float64
	BigFixedSizeChance float64
	RandomSize         bool
}

type Sample struct {
	URL              string
	ShootTimeSeconds float64
}

type GunConfig struct {
	Target string `validate:"required"`
}

type Gun struct {
	client geocounter_service.GeoCounterServiceV1Client
	conf   GunConfig
	aggr   core.Aggregator
	core.GunDeps
}

func NewGun(conf GunConfig) *Gun {
	return &Gun{conf: conf}
}

func (g *Gun) Bind(aggr core.Aggregator, deps core.GunDeps) error {
	conn, err := grpc.Dial(
		g.conf.Target,
		grpc.WithInsecure(),
		//nolint:SA1019
		grpc.WithTimeout(time.Second),
		grpc.WithUserAgent("load test, pandora custom shooter"))
	if err != nil {
		log.Fatalf("FATAL: %s", err)
	}
	g.client = geocounter_service.NewGeoCounterServiceV1Client(conn)
	g.aggr = aggr
	g.GunDeps = deps
	return nil
}

func (g *Gun) Shoot(ammo core.Ammo) {
	customAmmo := ammo.(*Ammo)
	g.shoot(customAmmo)
}

func (g *Gun) doReq(client geocounter_service.GeoCounterServiceV1Client, lowerLeftLat float64, lowerLeftLon float64, upperRightLat float64, upperRightLon float64) int {
	code := 0

	checkInDate := "2030-01-01"
	checkOutDate := "2030-01-02"
	ages := "88,88"

	request := geocounter_service.TGetCountsRequest{
		LowerLeftLat:  &lowerLeftLat,
		LowerLeftLon:  &lowerLeftLon,
		UpperRightLat: &upperRightLat,
		UpperRightLon: &upperRightLon,
		CheckInDate:   &checkInDate,
		CheckOutDate:  &checkOutDate,
		Ages:          &ages,
	}
	out, err := client.GetCounts(context.TODO(), &request)

	if err != nil {
		log.Printf("FATAL: %s", err)
		code = 500
	}

	if out != nil {
		if out.GetError() != nil {
			code = 500
		} else {
			code = 200
		}
	}
	return code
}

func (g *Gun) simpleMethod(client geocounter_service.GeoCounterServiceV1Client, ammo *Ammo) int {
	return g.doReq(client, ammo.LowerLeftLat, ammo.LowerLeftLon, ammo.UpperRightLat, ammo.UpperRightLon)
}

func (g *Gun) randomMethod(client geocounter_service.GeoCounterServiceV1Client, ammo *Ammo) int {
	lowerLeftLat := rand.Float64()*360 - 180
	lowerLeftLon := rand.Float64()*180 - 90
	upperRightLat := 0.
	upperRightLon := 0.
	if ammo.RandomSize {
		upperRightLat = rand.Float64()*(180-lowerLeftLat) + lowerLeftLat
		upperRightLon = rand.Float64()*(90-lowerLeftLon) + lowerLeftLon
	} else {
		if ammo.BigFixedSizeChance > 1e-4 && rand.Float64() < ammo.BigFixedSizeChance {
			upperRightLat = lowerLeftLat + ammo.BigFixedSizeChance
			upperRightLon = lowerLeftLon + ammo.BigFixedSizeChance
		} else {
			upperRightLat = lowerLeftLat + ammo.FixedSize
			upperRightLon = lowerLeftLon + ammo.FixedSize
		}
	}

	return g.doReq(client, lowerLeftLat, lowerLeftLon, upperRightLat, upperRightLon)
}

func (g *Gun) pingMethod(client geocounter_service.GeoCounterServiceV1Client, ammo *Ammo) int {
	code := 0

	request := pb.TPingRpcReq{}
	out, err := client.Ping(context.TODO(), &request)

	if err != nil {
		log.Printf("FATAL: %s", err)
		code = 500
	}

	if out != nil {
		code = 200
	}
	return code
}

func (g *Gun) shoot(ammo *Ammo) {
	code := 0
	sample := netsample.Acquire(ammo.Tag)

	switch ammo.Tag {
	case "/Simple":
		code = g.simpleMethod(g.client, ammo)
	case "/Random":
		code = g.randomMethod(g.client, ammo)
	case "/RandomBig":
		code = g.randomMethod(g.client, ammo)
	case "/Ping":
		code = g.pingMethod(g.client, ammo)
	default:
		code = 404
	}

	defer func() {
		sample.SetProtoCode(code)
		g.aggr.Report(sample)
	}()
}

func main() {
	fs := coreimport.GetFs()
	coreimport.Import(fs)

	coreimport.RegisterCustomJSONProvider("custom_provider", func() core.Ammo { return &Ammo{} })

	register.Gun("my_custom_gun_name", NewGun, func() GunConfig {
		return GunConfig{
			Target: "default target",
		}
	})

	cli.Run()
}
