from argparse import ArgumentParser
from typing import Iterable

from lxml import etree
from yt.wrapper import YtClient

from travel.library.python.tools import replace_args_from_env
from travel.library.python.s3_client import S3Client
from travel.hotels.lib.python3.yql import yqllib


ROOT_ELEMENT = '''
<listings
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xsi:noNamespaceSchemaLocation="http://www.gstatic.com/localfeed/local_feed.xsd"
/>
'''


class Hotel:
    def __init__(self, row):
        self.slug = row['main_slug']
        self.name = row['name']
        self.address = row['original_address']
        self.country = row['country_code']
        self.latitude = row['lat']
        self.longitude = row['lon']
        self.phones = row['phones']

    def to_xml_element(self):
        hotel = etree.Element('listing')
        etree.SubElement(hotel, 'id').text = self.slug
        etree.SubElement(hotel, 'name').text = self.name

        etree.SubElement(hotel, 'address').text = self.address

        etree.SubElement(hotel, 'country').text = self.country
        etree.SubElement(hotel, 'latitude').text = str(self.latitude)
        etree.SubElement(hotel, 'longitude').text = str(self.longitude)
        if self.phones:
            etree.SubElement(hotel, 'phone', attrib={'type': 'main'}).text = self.phones[0]

        return hotel


class HotelsListGenerator(object):
    def __init__(self, args):
        self.yql_client = yqllib.create_client(token=args.yql_token, db=args.yt_proxy)
        self.yt_client = YtClient(proxy=args.yt_proxy, config={'token': args.yt_token})
        self.hotel_info_table = args.yt_hotel_info_table
        self.result_table = args.yt_result_table
        self.permalink_to_slug_table = args.yt_permalink_to_slug

        self.s3_client = S3Client(
            endpoint=args.s3_endpoint,
            bucket=args.s3_bucket,
            access_key=args.s3_access_key,
            access_secret_key=args.s3_access_secret_key,
        )
        self.s3_result_path = args.s3_result_path

    def read_hotel_info(self) -> Iterable[Hotel]:
        yield from map(Hotel, self.yt_client.read_table(self.result_table))

    def collect_hotels(self, resource_name, project_name, parameters):
        yqllib.run_yql_file(
            client=self.yql_client,
            resource_name=resource_name,
            project_name=project_name,
            parameters=parameters,
            sync=True
        )

    def build_xml(self):
        root = etree.fromstring(ROOT_ELEMENT)
        etree.SubElement(root, 'language').text = 'ru'

        for hotel in self.read_hotel_info():
            root.append(hotel.to_xml_element())
        return root

    def save_result(self, root):
        hotels_list_xml = etree.tostring(root, pretty_print=True, xml_declaration=True, encoding='UTF-8')
        self.s3_client.write(self.s3_result_path, hotels_list_xml)

    def run(self):
        self.collect_hotels(
            resource_name='collect_travelline_moscow_hotels.yql',
            project_name='Build Travelline Moscow',
            parameters={
                '$hotel_info_table': self.hotel_info_table,
                '$permalink_to_slug_table': self.permalink_to_slug_table,
                '$result_table': self.result_table
            }
        )
        self.save_result(self.build_xml())


def main():
    parser = ArgumentParser()
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-hotel-info-table', required=True)
    parser.add_argument('--yt-result-table', required=True)
    parser.add_argument('--yt-permalink-to-slug', required=True)
    parser.add_argument('--s3-endpoint', required=True)
    parser.add_argument('--s3-bucket', required=True)
    parser.add_argument('--s3-access_key', required=True)
    parser.add_argument('--s3-access_secret_key', required=True)
    parser.add_argument('--s3-result-path', required=True)

    args = parser.parse_args(args=replace_args_from_env())
    HotelsListGenerator(args).run()


if __name__ == '__main__':
    main()
