#include <travel/hotels/lib/cpp/encryption/codec.h>
#include <travel/hotels/lib/cpp/label/label.h>
#include <travel/hotels/proto2/label.pb.h>

#include <library/cpp/uri/uri.h>
#include <library/cpp/getopt/opt.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <library/cpp/protobuf/json/proto2json.h>


TString GetLabelFromRedirUrl(const TString& url) {
    static const NUri::TParseFlags URL_PARSE_FLAGS =
          ( NUri::TFeature::FeaturesRecommended
          | NUri::TFeature::FeatureConvertHostIDN
          | NUri::TFeature::FeatureEncodeExtendedDelim
          | NUri::TFeature::FeatureEncodePercent
          ) & ~NUri::TFeature::FeatureHashBangToEscapedFragment;

    NUri::TUri uri;
    if (uri.Parse(url, URL_PARSE_FLAGS) != NUri::TState::EParsed::ParsedOK) {
        return url;
    }
    TCgiParameters cgiParams(uri.GetField(NUri::TField::FieldQuery));
    TString label = cgiParams.Get("ProtoLabel");
    if (!label) {
        throw yexception() << "No label in url";
    }
    return label;
}

void DecodeAndPrintLabel(const NTravel::NLabel::TLabelCodec& codec, const TString& protoLabel, const NTravel::NLabel::TLabelCodec::TOpts& opts) {
    NTravelProto::TLabel label;
    codec.DecodeToMessage(protoLabel, &label, opts);
    auto json = NProtobufJson::Proto2Json(label,
                NProtobufJson::TProto2JsonConfig()
                    .SetFormatOutput(true)
                    .SetEnumMode(NProtobufJson::TProto2JsonConfig::EnumName)
                    .SetMapAsObject(true)
                    );
    Cout << json << Endl;
}

int main(int argc, const char* argv[]) {
    TString original, labelKeyPath;
    bool encLabel, decLabel, decProtoLabel, decProtoLabelWithCheckSum, decRedirUrl, decUrl, encUrl;
    {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption(0, "encode-label", "Do Encode label (with checksum, no prefix)")
            .NoArgument()
            .SetFlag(&encLabel);
        opts.AddLongOption(0, "decode-label", "Do Decode label (with checksum, no prefix)")
            .NoArgument()
            .SetFlag(&decLabel);
        opts.AddLongOption(0, "decode-proto-label", "Do Decode and print Protobuf label (no checksum)")
            .NoArgument()
            .SetFlag(&decProtoLabel);
        opts.AddLongOption(0, "decode-proto-label-with-checksum", "Do Decode Protobuf With Checksum")
            .NoArgument()
            .SetFlag(&decProtoLabelWithCheckSum);
        opts.AddLongOption(0, "encode-url", "Do Encode url")
            .NoArgument()
            .SetFlag(&encUrl);
        opts.AddLongOption(0, "decode-url", "Do Decode url")
            .NoArgument()
            .SetFlag(&decUrl);
        opts.AddLongOption('u', "decode-redir-url", "Do Decode Redir Url")
            .NoArgument()
            .SetFlag(&decRedirUrl);
        opts.AddLongOption('l', "labelkey-path", "Label Key Path")
            .StoreResult(&labelKeyPath);
        opts.SetFreeArgsNum(1);
        opts.SetFreeArgTitle(0, "Input string");
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
        original = res.GetFreeArgs().at(0);
    }
    using NTravel::NLabel::TLabelCodec;
    TLabelCodec codec;

    THolder<NTravel::NEncryption::TUrlCodec> urlCodec;
    if (labelKeyPath) {
        urlCodec.Reset(new NTravel::NEncryption::TUrlCodec(labelKeyPath));
    }

    if (decLabel) {
        Cout << codec.Decode(original, TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false)) << Endl;
    }
    if (encLabel) {
        Cout << codec.Encode(original, TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false)) << Endl;
    }
    if (decProtoLabel) {
        DecodeAndPrintLabel(codec, original, TLabelCodec::TOpts().WithCheckSum(false).WithPrefix(false));
    }
    if (decProtoLabelWithCheckSum) {
        DecodeAndPrintLabel(codec, original, TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false));
    }
    if (decRedirUrl) {
        DecodeAndPrintLabel(codec, GetLabelFromRedirUrl(original), TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false));
    }
    if (encUrl) {
        if (!urlCodec) {
            Cerr << "Labelkey should be specified" << Endl;
            return 1;
        }
        Cout << urlCodec->Encode(original) << Endl;
    }
    if (decUrl) {
        if (!urlCodec) {
            Cerr << "Labelkey should be specified" << Endl;
            return 1;
        }
        Cout << urlCodec->Decode(original) << Endl;
    }
    return 0;
}
