#include <travel/hotels/lib/cpp/encryption/codec.h>
#include <travel/hotels/lib/cpp/http/http_service.h>
#include <travel/hotels/lib/cpp/label/label.h>

#include <library/cpp/getopt/opt.h>
#include <library/cpp/sighandler/async_signals_handler.h>

#include <util/system/event.h>

using namespace NTravel;

class TService {
private:
    NEncryption::TUrlCodec UrlCodec_;
    NLabel::TLabelCodec  LabelCodec_;
    NEncryption::TTokenCodec  TokenCodec_;
    NHttp::TServer Http_;
    TAutoEvent            StopEvent_;

private:
    void OnEncodeUrl(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(UrlCodec_.Encode(httpReq.Query().Get("value"))));
    }
    void OnDecodeUrl(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(UrlCodec_.Decode(httpReq.Query().Get("value"))));
    }
    void OnEncodeLabel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(LabelCodec_.Encode(httpReq.Query().Get("value"), NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false))));
    }
    void OnDecodeLabel(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(LabelCodec_.Decode(httpReq.Query().Get("value"), NLabel::TLabelCodec::TOpts().WithCheckSum(true).WithPrefix(false))));
    }
    void OnEncodeToken(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(TokenCodec_.Encode(httpReq.Query().Get("value"))));
    }
    void OnDecodeToken(const NHttp::TRequest& httpReq, const NHttp::TOnResponse& responseCb) {
        responseCb(THttpResponse().SetContent(TokenCodec_.Decode(httpReq.Query().Get("value"))));
    }
public:
    TService(const TString& host, int port, const TString& labelKeyPath, const TString& tokenKeyPath)
        : UrlCodec_(NEncryption::TUrlCodec(labelKeyPath))
        , TokenCodec_(NEncryption::TTokenCodec(tokenKeyPath))
    {
        Http_.AddHandler("/encode_url", NHttp::ExternalWithoutTvm(), this, &TService::OnEncodeUrl);
        Http_.AddHandler("/decode_url", NHttp::ExternalWithoutTvm(), this, &TService::OnDecodeUrl);
        Http_.AddHandler("/encode_label", NHttp::ExternalWithoutTvm(), this, &TService::OnEncodeLabel);
        Http_.AddHandler("/decode_label", NHttp::ExternalWithoutTvm(), this, &TService::OnDecodeLabel);
        Http_.AddHandler("/encode_token", NHttp::ExternalWithoutTvm(), this, &TService::OnEncodeToken);
        Http_.AddHandler("/decode_token", NHttp::ExternalWithoutTvm(), this, &TService::OnDecodeToken);
        NTravelProto::NAppConfig::TConfigHttp cfg;
        cfg.SetHost(host);
        cfg.SetPort(port);
        Http_.Start(cfg);
    }
    void Run() {
        StopEvent_.Wait();
    }
    void Stop() {
        Http_.Stop();
        StopEvent_.Signal();
    }
};

int main(int argc, const char* argv[]) {
    TString host, labelKeyPath, tokenKeyPath;
    int port;
    {
        NLastGetopt::TOpts opts;
        opts.AddHelpOption('h');
        opts.AddLongOption(  0, "host", "Listen host")
            .DefaultValue("localhost")
            .StoreResult(&host);
        opts.AddLongOption('p', "port", "Listen port")
            .DefaultValue(8080)
            .StoreResult(&port);
        opts.AddLongOption('l', "labelkey-path", "Label Key Path")
            .StoreResult(&labelKeyPath)
            .Required();
        opts.AddLongOption('t', "tokenkey-path", "Token Key Path")
            .StoreResult(&tokenKeyPath)
            .Required();
        NLastGetopt::TOptsParseResult res(&opts, argc, argv);
    }
    {
        TService svc(host, port, labelKeyPath, tokenKeyPath);
        for (int signal: {SIGINT, SIGTERM}) {
            SetAsyncSignalFunction(signal, [&svc](int) {
                svc.Stop();
            });
        }
        svc.Run();
    }
    return 0;
}
