package main

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"github.com/spf13/cobra"
	excelize "github.com/xuri/excelize/v2"
	"net/url"
	"os"
	"reflect"
	"strconv"
	"sync"
)

const altayURL = "https://altay-unification-service-2.common-ext.yandex-team.ru/unify"

var ytToken string

type address struct {
	Precision string `json:"precision"`
	OneLine   string `json:"one_line"`
}

type unificationResponse struct {
	Success bool      `json:"success"`
	Address []address `json:"address"`
}

type result struct {
	ID                  string
	RegionCode          string
	Region              string
	Name                string
	FullName            string
	UnifiedAddress      string
	RawAddress          string
	ExactAddress        bool
	INN                 string
	OGRN                string
	LegalName           string
	AccrediterName      string
	AccrediterINN       string
	AccreditationNumber string
	AccreditationDate   string
	AccreditationValid  string
	Stars               string
	RoomNumber          int
}

type permalinkedResult struct {
	RoomNumber             int
	ID                     string
	OGRN                   string
	Name                   string
	FullName               string
	INN                    string
	RegionCode             string
	Region                 string
	LegalName              string
	Address                string
	Permalink              string
	RequiresClusterization bool
	Comment                string
}

type whiteListRecord struct {
	OriginalID string `yson:"OriginalId"`
	PartnerID  string `yson:"PartnerId"`
	MirID      string `yson:"MirId"`
}

func unifyAddress(row []string, logger log.Logger) *result {
	res := &result{
		ID:                  row[0],
		RegionCode:          row[1],
		Region:              row[2],
		Name:                row[3],
		FullName:            row[4],
		RawAddress:          row[5],
		INN:                 row[6],
		OGRN:                row[7],
		LegalName:           row[8],
		AccrediterName:      row[9],
		AccrediterINN:       row[10],
		AccreditationNumber: row[11],
		AccreditationDate:   row[12],
		AccreditationValid:  row[13],
	}
	if len(row) > 14 {
		res.Stars = row[14]
	}
	if len(row) > 15 {
		rooms, err := strconv.Atoi(row[15])
		if err == nil {
			res.RoomNumber = rooms
		}
	}
	getURL := fmt.Sprintf("%s?address=%s", altayURL, url.QueryEscape(res.RawAddress))
	var response unificationResponse
	var exact bool
	_, err := resty.New().R().SetResult(&response).Get(getURL)
	if err != nil {
		logger.Errorf("Unable to unify address '%s': %v", res.RawAddress, err)
		return res
	}
	if !response.Success {
		logger.Errorf("Unable to unify address '%s'", res.RawAddress)
		return res
	}
	if len(response.Address) != 1 {
		logger.Errorf("Unexpected amount of addresses")
		return res
	}
	if response.Address[0].Precision != "EXACT" {
		logger.Warnf("Bad precision %s for address '%s'", response.Address[0].Precision, res.RawAddress)
		exact = false
	} else {
		exact = true
	}
	res.ExactAddress = exact
	res.UnifiedAddress = response.Address[0].OneLine
	return res
}

func merge(ctx context.Context, chans []chan *result) <-chan *result {
	var wg sync.WaitGroup
	out := make(chan *result)
	output := func(c <-chan *result) {
		defer wg.Done()
		for n := range c {
			select {
			case out <- n:
				continue
			case <-ctx.Done():
				return
			}
		}
	}
	wg.Add(len(chans))
	for _, c := range chans {
		go output(c)
	}
	go func() {
		wg.Wait()
		close(out)
	}()
	return out
}

func unifyAddressStream(ctx context.Context, incoming <-chan []string, n int, logger log.Logger) <-chan *result {
	var chans = make([]chan *result, n)
	for i := 0; i < n; i++ {
		ch := make(chan *result)
		chans[i] = ch
		go func(out chan<- *result) {
			defer close(out)
			for row := range incoming {
				u := unifyAddress(row, logger)
				if u == nil {
					continue
				}
				select {
				case out <- u:
					continue
				case <-ctx.Done():
					return
				}
			}
		}(ch)
	}
	return merge(ctx, chans)
}

func readExcel(ctx context.Context, filename string, sheetname string) (<-chan []string, error) {
	res := make(chan []string)
	f, err := excelize.OpenFile(filename)
	if err != nil {
		return nil, err
	}
	rows, err := f.GetRows(sheetname)
	if err != nil {
		return nil, err
	}
	go func() {
		defer close(res)
		for _, row := range rows[1:] {
			select {
			case res <- row:
				continue
			case <-ctx.Done():
				return
			}
		}
	}()
	return res, nil
}

func uploadKsr() {
	logger, _ := zap.New(zap.ConsoleConfig(log.InfoLevel))
	ch, err := readExcel(context.Background(), "База КСР.xlsx", "all_ksr")
	if err != nil {
		panic(err)
	}
	var results []*result
	for r := range unifyAddressStream(context.Background(), ch, 16, logger) {
		results = append(results, r)
	}
	err = ytstorage.Save(context.Background(), results, "//home/travel/tivelkov/mir/ksr-base", 1, ytToken, "hahn")
	if err != nil {
		panic(err)
	}
}

func uploadMatched() {
	ch, err := readExcel(context.Background(), "permalinks.xlsx", "Лист1")
	if err != nil {
		panic(err)
	}
	var results []*permalinkedResult
	for row := range ch {
		roomNum, _ := strconv.Atoi(row[0])
		res := &permalinkedResult{
			RoomNumber: roomNum,
			ID:         row[1],
			OGRN:       row[2],
			Name:       row[3],
			FullName:   row[4],
			INN:        row[5],
			RegionCode: row[6],
			Region:     row[7],
			LegalName:  row[8],
		}
		if len(row) > 9 {
			res.Address = row[9]
		}
		if len(row) > 10 {
			res.Permalink = row[10]
		}
		if len(row) > 11 {
			res.RequiresClusterization = row[11] == "да"
		}
		if len(row) > 12 {
			res.Comment = row[12]
		}
		results = append(results, res)
	}
	err = ytstorage.Save(context.Background(), results, "//home/travel/tivelkov/mir/ksr-matched", 1, ytToken, "hahn")
	if err != nil {
		panic(err)
	}
}

func download() {
	downloadTable("//home/travel/tivelkov/mir/ksr-whitelist-combined")
}

func downloadTable(table string) {
	data, err := ytstorage.Load(context.Background(), reflect.TypeOf(whiteListRecord{}), table, ytToken, "hahn")
	if err != nil {
		panic(err)
	}
	for _, d := range data {
		wlr := d.(*whiteListRecord)
		fmt.Printf("MirListHotel { OriginalId: \"%s\"; PartnerId: %s; MirId: \"%s\"; Enabled: true }\r\n",
			wlr.OriginalID, wlr.PartnerID, wlr.MirID)
	}
}

func generateFromModeratedExcel() {
	ch, err := readExcel(context.Background(), "mir_popular.xlsx", "Sheet1")
	if err != nil {
		panic(err)
	}
	for row := range ch {
		if len(row) < 6 {
			continue
		}

		oID := row[1]
		pID := row[3]
		mID := row[5]
		e := row[5] != ""
		if !e {
			mID = "0"
		}
		name := row[4]
		fmt.Printf("MirListHotel { OriginalId: \"%s\"; PartnerId: %s; MirId: \"%s\"; Enabled: %t }  # %s\r\n",
			oID, pID, mID, e, name)
	}
}

func main() {
	token, exists := os.LookupEnv("YT_TOKEN")
	if !exists {
		panic("no YT_TOKEN env var")
	}
	ytToken = token
	var rootCmd = &cobra.Command{Use: "mir-whitelist-parser"}
	var cmdUploadKsr = &cobra.Command{Use: "upload-ksr", Run: func(cmd *cobra.Command, args []string) {
		uploadKsr()
	}}
	var cmdUploadMatched = &cobra.Command{Use: "upload-matched", Run: func(cmd *cobra.Command, args []string) {
		uploadMatched()
	}}
	var cmdDownload = &cobra.Command{Use: "download", Run: func(cmd *cobra.Command, args []string) {
		download()
	}}
	var cmdGenerateFromModerated = &cobra.Command{Use: "gen-from-moderated", Run: func(cmd *cobra.Command, args []string) {
		generateFromModeratedExcel()
	}}
	rootCmd.AddCommand(cmdUploadKsr, cmdDownload, cmdUploadMatched, cmdGenerateFromModerated)
	_ = rootCmd.Execute()
}
