# -*- coding: utf-8 -*-

from argparse import ArgumentParser
import logging
import requests
import time
from datetime import datetime
from dateutil import relativedelta

from yql.api.v1.client import YqlClient

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt import ytlib
from travel.hotels.lib.python3.yt.versioned_path import VersionedPath


class Surface:
    def __init__(self, name, query_suffix, counter_id, is_mobile):
        self.name = name
        self.query_suffix = query_suffix
        self.counter_id = counter_id
        self.is_mobile = is_mobile


class Runner(object):
    SurfacePortal = Surface('portal', '_portal', 50912507, None)
    SurfaceRaspDesktop = Surface('rasp_desktop', '_rasp', 99704, False)
    SurfaceRaspMobile = Surface('rasp_mobile', '_rasp', 22352497, True)

    SCRIPTS = [
        ('avia', ['avia_order_profit', 'avia_order_confirmed'], [SurfacePortal, SurfaceRaspDesktop, SurfaceRaspMobile]),
        ('buses', ['bus_order_profit', 'bus_order_confirmed'], [SurfacePortal, SurfaceRaspDesktop, SurfaceRaspMobile]),
        ('trains', ['zhd_order_profit', 'zhd_order_confirmed'], [SurfacePortal, SurfaceRaspDesktop, SurfaceRaspMobile]),
        ('hotels', ['hotels_bookings_offline'], [SurfacePortal]),   # Debug target is hotels_bookings_debug
        ('tours', ['tours_bookings_offline'], [SurfacePortal]),
        ('hotels_boy', ['hotels_bookings_boy_offline'], [SurfacePortal]),
    ]

    def __init__(self, args):
        assert args.yql_token or args.yql_token_path
        yql_args = {
            'db': args.yt_proxy,
            'token': args.yql_token,
            'token_path': args.yql_token_path,
        }
        assert args.yt_token or args.yt_token_path
        self.yql_client = YqlClient(**yql_args)
        yt_config = {
            'token': args.yt_token,
            'token_path': args.yt_token_path,
        }
        self.yt_client = ytlib.create_client(proxy=args.yt_proxy, config=yt_config)
        self.args = args

    def run(self):
        with self.yt_client.Transaction() as t:
            with VersionedPath(self.args.working_path, yt_client=self.yt_client) as working_path:
                logging.info(f'Going to run at {working_path}')
                for script in self.SCRIPTS:
                    for target in script[1]:
                        for surface in script[2]:
                            self.run_script(working_path, t, script[0], target, surface)

    def run_script(self, working_path, transaction, script_name, target, surface):
        logging.info(f"Running script {script_name}, target {target}, surface {surface.name}")
        table_prefix = f'{script_name}_{surface.name}_{target}'
        query_args = {
            '$output_path': ytlib.join(working_path, table_prefix),
            '$report_date': self.args.report_date,
            '$target': target,
            '$counter_id': surface.counter_id,
        }
        if surface.is_mobile is not None:
            query_args['$is_mobile'] = surface.is_mobile
        yqllib.run_yql_file(self.yql_client, script_name + surface.query_suffix + '.yql', 'orders_to_metrika',
                            parameters=query_args, transaction_id=transaction.transaction_id)
        self.send_table_to_metrika(surface.counter_id, working_path, table_prefix, 'YCLID', 'Yclid')
        self.send_table_to_metrika(surface.counter_id, working_path, table_prefix, 'CLIENT_ID', 'ClientID')
        self.send_table_to_metrika(surface.counter_id, working_path, table_prefix, 'USER_ID', 'UserID')

    def send_table_to_metrika(self, counter_id, working_path, table_prefix, client_id_type, key_column):
        table_name = f'{table_prefix}_{key_column}'
        table_fullname = ytlib.join(working_path, table_name)
        if not self.yt_client.exists(table_fullname):
            logging.info(f'No table {table_name}, nothing to send')
            return
        tsv_lines = []
        for row in self.yt_client.read_table(table_fullname):
            tsv_lines.append("{},{},{},{},{}".format(row[key_column], row['Target'], row['DateTime'],
                                                     row['Price'], row['Currency']))
        logging.info(f"Got {len(tsv_lines)} records for client_id_type = {client_id_type}")
        if not tsv_lines:
            logging.info(f'Empty table, nothing to send for client_id_type = {client_id_type}')
            return
        tsv_lines = [f'{key_column},Target,DateTime,Price,Currency'] + tsv_lines
        url = f"https://api-metrika.yandex.net/management/v1/counter/{counter_id}/offline_conversions/upload"
        params = {
            'client_id_type': client_id_type,
            'comment': table_name
        }
        headers = {
            "Authorization": f"OAuth {self.args.metrika_token}"
        }
        files = {
            'file': '\n'.join(tsv_lines)
        }
        logging.info("Sending...")
        logging.info("Going to send")
        logging.info('\n'.join(tsv_lines))

        retries = 5
        while True:
            try:
                rsp = requests.post(url, params=params, headers=headers, files=files)
                rsp.raise_for_status()
                logging.info(f"Success: {rsp.text}")
                break
            except Exception as e:
                logging.error(f'Attempt failed, retries left: {retries}', e)
                retries -= 1
                if retries == 0:
                    raise
                time.sleep(10)  # Э, нет, торопиться не надо, торопиться не надо


def main():
    def get_default_report_date():
        return (datetime.today() - relativedelta.relativedelta(days=1)).strftime('%Y-%m-%d')

    FORMAT = '%(asctime)-15s | %(levelname)-4.4s | %(name)-12.12s | %(message)s'
    logging.basicConfig(level=logging.INFO, format=FORMAT)
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yql-token')
    parser.add_argument('--yql-token-path')
    parser.add_argument('--yt-token')
    parser.add_argument('--yt-token-path')
    parser.add_argument('--metrika-token', required=True)
    parser.add_argument('--report-date', type=str, default=get_default_report_date())
    parser.add_argument('--working-path', required=True)
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()
