import datetime
import logging
import sys
from argparse import ArgumentParser
from yt.wrapper import YtClient

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yt import ytlib

from travel.hotels.tools.page_speed_data_downloader.page_speed_client import PageSpeedClient
from travel.hotels.tools.page_speed_data_downloader.urls import URLS

LOG = logging.getLogger(__name__)

STRING_FIELDS_FOR_REPORT = {
    'lighthouseResult.audits.first-contentful-paint.displayValue': 'string',
    'lighthouseResult.audits.speed-index.displayValue': 'string',
    'lighthouseResult.audits.cumulative-layout-shift.displayValue': 'string',
    'lighthouseResult.audits.largest-contentful-paint.displayValue': 'string',
    'lighthouseResult.audits.largest-contentful-paint-element.displayValue': 'string',
    'lighthouseResult.audits.interactive.displayValue': 'string',

    'loadingExperience.metrics.FIRST_CONTENTFUL_PAINT_MS.category': 'string',
    'loadingExperience.metrics.FIRST_INPUT_DELAY_MS.category': 'string',
    'loadingExperience.metrics.CUMULATIVE_LAYOUT_SHIFT_SCORE.category': 'string',
    'loadingExperience.metrics.LARGEST_CONTENTFUL_PAINT_MS.category': 'string',

    'originLoadingExperience.metrics.FIRST_CONTENTFUL_PAINT_MS.category': 'string',
    'originLoadingExperience.metrics.FIRST_INPUT_DELAY_MS.category': 'string',
    'originLoadingExperience.metrics.CUMULATIVE_LAYOUT_SHIFT_SCORE.category': 'string',
    'originLoadingExperience.metrics.LARGEST_CONTENTFUL_PAINT_MS.category': 'string',
}

NUMERICAL_FIELDS_FOR_REPORT = {
    # метрики LightHouse
    'lighthouseResult.audits.first-contentful-paint.numericValue': 'double',
    'lighthouseResult.audits.speed-index.numericValue': 'double',
    'lighthouseResult.audits.cumulative-layout-shift.numericValue': 'double',
    'lighthouseResult.audits.largest-contentful-paint.numericValue': 'double',
    'lighthouseResult.audits.largest-contentful-paint-element.numericValue': 'double',
    'lighthouseResult.audits.interactive.numericValue': 'double',

    # общий перформанс
    'lighthouseResult.categories.performance.score': 'double',

    # стандартные метрики Google PageSpeed
    'loadingExperience.metrics.FIRST_CONTENTFUL_PAINT_MS.percentile': 'double',
    'loadingExperience.metrics.FIRST_INPUT_DELAY_MS.percentile': 'double',
    'loadingExperience.metrics.CUMULATIVE_LAYOUT_SHIFT_SCORE.percentile': 'double',
    'loadingExperience.metrics.LARGEST_CONTENTFUL_PAINT_MS.percentile': 'double',

    'originLoadingExperience.metrics.FIRST_CONTENTFUL_PAINT_MS.percentile': 'double',
    'originLoadingExperience.metrics.FIRST_INPUT_DELAY_MS.percentile': 'double',
    'originLoadingExperience.metrics.CUMULATIVE_LAYOUT_SHIFT_SCORE.percentile': 'double',
    'originLoadingExperience.metrics.LARGEST_CONTENTFUL_PAINT_MS.percentile': 'double',
}

SERVICE_FIELDS = {
    'id': 'string',
    'type': 'string',
    'strategy': 'string',
    'url_API': 'string',
    'url_orig': 'string',
    'url_orig_domain': 'string',
    'url_group': 'string',
    'url_details': 'string',
    'captchaResult': 'string',
}

FIELDS_FOR_REPORT = {**SERVICE_FIELDS, **NUMERICAL_FIELDS_FOR_REPORT, **STRING_FIELDS_FOR_REPORT}


def format_name(name):
    return name.lower().replace('.', '_').replace('-', '_')


def patch_type(value):
    if type(value) is int:
        return float(value)
    return value


TABLE_SCHEMA = ytlib.schema_from_dict({
    **{format_name(x): y for x, y in FIELDS_FOR_REPORT.items()},
    'fielddate': 'string'
})


class Runner:
    def __init__(self, args):
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)
        self.yt_token = args.yt_token
        self.yt_path = args.yt_path
        self.page_speed_client = PageSpeedClient(args.page_speed_token)
        self.append_results = args.append_results
        self.args = args

    def run(self):
        now = datetime.datetime.utcnow()
        timestamp = now.strftime('%Y-%m-%d %H:%M:%S')

        urls = URLS if self.args.urls_limit is None else URLS[:self.args.urls_limit]
        results = self.page_speed_client.get_results(urls, strategies=['mobile', 'desktop'],
                                                     fields_for_report=FIELDS_FOR_REPORT.keys())
        patched_results = [Runner._patch_single_result(x, timestamp) for x in results]

        if self.append_results:
            self.yt_client.create('table', self.yt_path, recursive=True, ignore_existing=True, attributes={"schema": TABLE_SCHEMA})
            self.yt_client.write_table(self.yt_client.TablePath(self.yt_path, append=True), patched_results)
        else:
            table_path = ytlib.join(self.yt_path, now.strftime('%Y-%m-%dT%H:%M:%SZ'))
            self.yt_client.create('table', table_path, recursive=True, ignore_existing=True, attributes={"schema": TABLE_SCHEMA})
            self.yt_client.write_table(table_path, patched_results)

    @staticmethod
    def _patch_single_result(result, date):
        patched = {format_name(k): patch_type(v) for k, v in result.items()}
        patched['fielddate'] = date
        return patched


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
                        stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yt-path', required=True)
    parser.add_argument('--append-results', action="store_true", default=False)
    parser.add_argument('--page-speed-token', required=True)
    parser.add_argument('--urls-limit', type=int)
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()
