import multiprocessing.dummy as multiprocessing
import urllib
import random
import time
import requests

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


def requests_retry_session(requests_retries=10, requests_backoff_factor=0.3):
    """
    Замена для requests, для повторного подключения в случае ошибки
    Взято из статьи: https://www.peterbe.com/plog/best-practice-with-retries-with-requests
    :param requests_retries: число попыток
    :param requests_backoff_factor: фактор увеличения времени задержки между попытками. См. подробнее про backoff_factor:
    {backoff factor} * (2 ^ ({number of total retries} - 1))
    https://urllib3.readthedocs.io/en/latest/reference/urllib3.util.html#module-urllib3.util.retry
    """
    retry = Retry(
        total=requests_retries,
        read=requests_retries,
        connect=requests_retries,
        backoff_factor=requests_backoff_factor,
        status_forcelist=(500, 502, 504),
    )
    adapter = HTTPAdapter(max_retries=retry)
    session = requests.Session()
    session.mount('https://', adapter)
    return session


def flatten_data(data):
    for k, v in data.items():
        if type(v) in [int, float, str, bool] or v is None:
            yield (k, v)
        elif type(v) == list:
            if len([x for x in v if type(x) is not dict]) > 0:  # can't be flattened
                continue
            for item in v:
                yield from ((f'{k}.{sub_key}', sub_value) for sub_key, sub_value in flatten_data(item))
        elif type(v) == dict:
            yield from ((f'{k}.{sub_key}', sub_value) for sub_key, sub_value in flatten_data(v))
        else:
            raise Exception(f"Unsupported type {type(v)} (field {k}: {v})")


class PageSpeedClient:
    def __init__(self, token, request_processes=10, request_timeout=180, captcha_retries=3, captcha_backoff_factor=200):
        """
        :param token: api-ключ от сервиса Google PageSpeed
        :param request_processes: кол-во процессов для опроса сервиса, если больше, то можем уперется в органичение: Queries per 100 seconds per user 60
        Подробнее про лимиты
        https://console.developers.google.com/apis/api/pagespeedonline.googleapis.com/quotas?project=useful-theory-213214&duration=PT1H
        :param request_timeout: время ожидания для получения ответа от сервера, секунды
        :param captcha_retries: попыток повторить, если спрашивают CAPTCHA
        :param captcha_backoff_factor: фактор увеличения времени задержки между попытками подключиться в случае запроса
        """
        self.token = token
        self.request_processes = request_processes
        self.request_timeout = request_timeout
        self.captcha_retries = captcha_retries
        self.captcha_backoff_factor = captcha_backoff_factor
        self.pagespeed_url_pattern = 'https://www.googleapis.com/pagespeedonline/v5/runPagespeed?{params}&key={token}'

    def get_results(self, urls, strategies, fields_for_report):
        """
        :param urls: список URL'ов для опроса
        :param strategies: какие стратегии (в терминах PageSpeed) использовать при анализе указанных URL'ов
        :param fields_for_report: какие поля ответа вернуть
        """
        urls_for_api = self._generate_urls_for_api(urls_data=urls, strategies=strategies, token=self.token)
        random.shuffle(urls_for_api)

        p = multiprocessing.Pool(processes=self.request_processes)
        ps_results = p.map(lambda url: self._get_page_speed_result(url, fields_for_report), urls_for_api)
        p.close()
        p.join()

        return list(ps_results)

    def _generate_urls_for_api(self, urls_data, strategies, token):
        return [{
            'url_orig': url_data['url'],
            'url_group': url_data['group'],
            'strategy': strategy,
            'url_API': self.pagespeed_url_pattern.format(
                params=urllib.parse.urlencode({'url': url_data['url'], 'strategy': strategy}),
                token=token
            ),
            'url_orig_domain': url_data['domain'],
            'url_details': url_data['details'],
            'type': url_data['type']
        } for url_data in urls_data for strategy in strategies]

    def _get_page_speed_result(self, url_api, fields_for_report):
        for captcha_attempt in range(self.captcha_retries):
            try:
                response = requests_retry_session().get(
                    url=url_api['url_API'],
                    timeout=self.request_timeout
                )
            except Exception as x:
                result = {'id': 'Ошибка: ' + x.__class__.__name__}
            else:
                result = {k: v for k, v in flatten_data(response.json())}

            if (result.get('captchaResult') != 'CAPTCHA_NOT_NEEDED') and (result.get('captchaResult') is not None):
                delay = self.captcha_backoff_factor * (2 ** captcha_attempt)
                time.sleep(delay)
            else:
                result.update(url_api)
                final_result = {key: result.get(key) for key in fields_for_report}
                print('{}: \tОбработка завершена успешно'.format(url_api['url_orig']))
                return final_result

        return {}
