# -*- coding: utf-8 -*-

import logging
import csv
import io
import time
from typing import Dict, Tuple

from library.python import resource

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient, ypath_split

import travel.hotels.proto2.hotels_pb2 as hotels_pb2
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt.versioned_path import VersionedPath, StandardCleanupStrategy
from travel.hotels.lib.python3.yt import ytlib

LOG = logging.getLogger(__name__)

expedia_features_schema = ytlib.schema_from_dict({
    "expedia_id": "string",
    "expedia_en_name": "string",
    "expedia_ru_name": "string",
    "id": "string",
    "name": "string",
    "type": "string",
    "bool_value": "boolean",
    "value_id": "string",
    "value_name": "string",
    "display_value": "string",
    "top_feature_importance": "double",
    "category": "string",
    "category_name": "string",
    "icon_id": "string",
})

travelline_features_schema = ytlib.schema_from_dict({
    "travelline_category": "string",
    "travelline_id": "string",
    "travelline_name": "string",
    "id": "string",
    "name": "string",
    "type": "string",
    "bool_value": "boolean",
    "value_id": "string",
    "value_name": "string",
    "display_value": "string",
    "top_feature_importance": "double",
    "category": "string",
    "category_name": "string",
    "icon_id": "string",
})

bronevik_features_schema = ytlib.schema_from_dict({
    "bronevik_id": "string",
    "bronevik_en_name": "string",
    "bronevik_ru_name": "string",
    "id": "string",
    "name": "string",
    "type": "string",
    "bool_value": "boolean",
    "value_id": "string",
    "value_name": "string",
    "display_value": "string",
    "top_feature_importance": "double",
    "category": "string",
    "category_name": "string",
    "icon_id": "string",
})

expedia_beds_schema = ytlib.schema_from_dict({
    "expedia_type": "string",
    "expedia_size": "string",
    "type": "string",
    "is_hidden": "boolean",
    "nominative_case_singular": "string",
    "genitive_case_singular": "string",
    "genitive_case_plural": "string",
})

travelline_beds_schema = ytlib.schema_from_dict({
    "travelline_id": "string",
    "travelline_name": "string",
    "type": "string",
    "quantity": "uint32",
    "is_hidden": "boolean",
    "nominative_case_singular": "string",
    "genitive_case_singular": "string",
    "genitive_case_plural": "string",
})

bronevik_beds_schema = ytlib.schema_from_dict({
    "bronevik_type": "string",
    "type": "string",
    "quantity": "uint32",
    "is_hidden": "boolean",
    "nominative_case_singular": "string",
    "genitive_case_singular": "string",
    "genitive_case_plural": "string",
})


class FeatureValue:
    def __init__(self, top_feature_importance: float, bool_value: bool, value_id: str, value_name: str, display_value: str, icon_id: str):
        self.top_feature_importance: float = top_feature_importance
        self.bool_value: bool = bool_value
        self.value_id: str = value_id
        self.value_name: str = value_name
        self.display_value: str = display_value
        self.icon_id: str = icon_id


class Feature:
    def __init__(self, id: str, name: str, type: str, category: str, category_name: str):
        self.id: str = id
        self.name: str = name
        self.type: str = type
        self.category: str = category
        self.category_name: str = category_name
        self.values: Dict[Tuple, FeatureValue] = {}

    def add_value(self, value: FeatureValue):
        self.values[(value.bool_value, value.value_id)] = value


class FeatureReader():
    def __init__(self, skip_errors):
        self.skip_errors = skip_errors
        self.has_features_errors = False
        self.categories_db_resourse = 'all_categories.csv'
        self.categories_database = self._read_categories_database()
        self.features_db_resourse = 'all_features.csv'
        self.features_database = self._read_features_database()

    @staticmethod
    def _patch_csv_row(row):
        for name in row.keys():
            if row[name] == '':
                row[name] = None
        row['bool_value'] = {'true': True, 'false': False, None: None}[row['bool_value']]
        if row.get('top_feature_importance') is not None:
            row['top_feature_importance'] = float(row['top_feature_importance'])

    def _read_categories_database(self):
        raw_features_database = resource.find(self.categories_db_resourse).decode('utf-8')
        categories_database: Dict[str, str] = {}
        for x in csv.DictReader(io.StringIO(raw_features_database), delimiter=',', quotechar='"'):
            try:
                assert x['id'] != '', 'Category id is required'
                assert x['name'] != '', 'Category name is required'
                id = x['id']
                assert id not in categories_database, f'Duplicate category id: {id}'
                categories_database[id] = x['name']
            except Exception as e:
                if not self.skip_errors:
                    raise
                LOG.error(str(e))
                self.has_features_errors = True
        return categories_database

    def _read_features_database(self):
        raw_features_database = resource.find(self.features_db_resourse).decode('utf-8')
        seen_display_values = set()
        features_database: Dict[str, Feature] = {}
        for x in csv.DictReader(io.StringIO(raw_features_database), delimiter=',', quotechar='"'):
            try:
                FeatureReader._patch_csv_row(x)
                feauture_id = x['id']
                assert feauture_id != '', 'Feature id is required'
                if feauture_id not in features_database:
                    category = x['category']
                    assert category in self.categories_database, f'Unknown category "{category}" (id={feauture_id})'
                    category_name = self.categories_database[category]
                    features_database[feauture_id] = Feature(feauture_id, x['name'], x['type'], category, category_name)
                else:
                    assert features_database[feauture_id].name == x['name'], f'Feature name mismatch (id={feauture_id})'
                    assert features_database[feauture_id].type == x['type'], f'Feature type mismatch (id={feauture_id})'
                    assert features_database[feauture_id].category == x['category'], f'Feature category mismatch (id={feauture_id})'
                assert (x['value_id'] is None) == (x['value_name'] is None), \
                        f'value_id and value_name should be set simultaneously (id={feauture_id})'
                assert x['bool_value'] is None or x['value_id'] is None, \
                        f'Only one of bool_value and value_id can be used simultaneously (id={feauture_id})'
                assert (x['bool_value'], x['value_id']) not in features_database[feauture_id].values, f'Duplicate value (id={feauture_id})'
                if x['display_value'] is not None:
                    assert x['display_value'] not in seen_display_values, f'Duplicate display_value ({x["display_value"]})'
                    seen_display_values.add(x['display_value'])
                assert x['icon_id'] is not None, f'Found feature without icon_id (id={feauture_id})'
                features_database[feauture_id].add_value(FeatureValue(x['top_feature_importance'], x['bool_value'], x['value_id'], x['value_name'], x['display_value'], x['icon_id']))
            except Exception as e:
                if not self.skip_errors:
                    raise
                LOG.error(str(e))
                self.has_features_errors = True
        return features_database

    def read_features(self, resource_name):
        data = resource.find(resource_name).decode('utf-8')
        result = []
        for x in csv.DictReader(io.StringIO(data), delimiter=',', quotechar='"'):
            try:
                FeatureReader._patch_csv_row(x)
                feature_id = x['id']
                assert feature_id != '', 'Feature id is required'
                assert feature_id in self.features_database, f'Unknown feature id ({feature_id})'
                feature = self.features_database[feature_id]
                value = (x['bool_value'], x['value_id'])
                assert feature.type == x['type'], f'Feature type mismatch (id={feature_id})'
                assert value in feature.values, f'Unknown feature value (id={feature_id}): {value}'
                value_from_db = feature.values[value]
                x['name'] = feature.name
                x['value_name'] = value_from_db.value_name
                x['display_value'] = value_from_db.display_value
                x['top_feature_importance'] = value_from_db.top_feature_importance
                x['icon_id'] = value_from_db.icon_id
                x['category'] = feature.category
                x['category_name'] = feature.category_name
                result.append(dict(x))
            except Exception as e:
                if not self.skip_errors:
                    raise
                LOG.error(str(e))
                self.has_features_errors = True
        return result


class BedType:
    def __init__(self, type: str, is_hidden: bool, nominative_case_singular: str, genitive_case_singular: str, genitive_case_plural: str):
        self.type: str = type
        self.is_hidden: bool = is_hidden
        self.nominative_case_singular: str = nominative_case_singular
        self.genitive_case_singular: str = genitive_case_singular
        self.genitive_case_plural: str = genitive_case_plural


class BedsReader:
    def __init__(self, skip_errors):
        self.skip_errors = skip_errors
        self.has_errors = False
        self.beds_db_resourse = 'all_beds.csv'
        self.beds_database = self._read_beds_database()

    @staticmethod
    def _patch_csv_row(row):
        for k, v in row.items():
            assert v != '', f'Beds mapping column {k} has no value'
        if 'quantity' in row:
            row['quantity'] = int(row['quantity'])

    def _read_beds_database(self):
        raw_beds_database = resource.find(self.beds_db_resourse).decode('utf-8')
        beds_database: Dict[str, BedType] = {}
        for x in csv.DictReader(io.StringIO(raw_beds_database), delimiter=',', quotechar='"'):
            try:
                for col in ['type', 'is_hidden', 'nominative_case_singular', 'genitive_case_singular', 'genitive_case_plural']:
                    assert x[col] != '', f'{col} column is required'
                assert x['is_hidden'] in ['true', 'false'], 'is_hidden column must have value "true" or "false"'
                is_hidden = {'true': True, 'false': False}[x['is_hidden']]
                bed_type = x['type']
                assert bed_type not in beds_database, f'Duplicate bed type: {bed_type}'
                beds_database[bed_type] = BedType(bed_type, is_hidden, x['nominative_case_singular'], x['genitive_case_singular'], x['genitive_case_plural'])
            except Exception as e:
                if not self.skip_errors:
                    raise
                LOG.error(str(e))
                self.has_errors = True
        return beds_database

    def read_beds(self, resource_name):
        data = resource.find(resource_name).decode('utf-8')
        result = []
        for x in csv.DictReader(io.StringIO(data), delimiter=',', quotechar='"'):
            try:
                BedsReader._patch_csv_row(x)
                bed_type_id = x['type']
                assert bed_type_id in self.beds_database, f'Unknown bed type: {bed_type_id}'
                bed_type = self.beds_database[bed_type_id]
                res = dict(x)
                res.update({
                    'is_hidden': bed_type.is_hidden,
                    'nominative_case_singular': bed_type.nominative_case_singular,
                    'genitive_case_singular': bed_type.genitive_case_singular,
                    'genitive_case_plural': bed_type.genitive_case_plural,
                })
                result.append(res)
            except Exception as e:
                if not self.skip_errors:
                    raise
                LOG.error(str(e))
                self.has_errors = True
        return result


class Runner:
    def __init__(self, args):
        self.yql_client = YqlClient(db=args.yt_proxy, token=args.yql_token)
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)
        self.args = args
        self.feature_reader = FeatureReader(skip_errors=False)
        self.beds_reader = BedsReader(skip_errors=False)

    def run(self):
        versioned_path = VersionedPath(self.args.yt_path, yt_client=self.yt_client, cleanup_strategy=StandardCleanupStrategy(90, True))
        with versioned_path as work_path:
            self._run_process(work_path)
            # copy results to another cluster
            if self.args.transfer_results:
                versioned_path.transfer_results(self.args.transfer_destination, self.args.yt_token, self.args.yt_proxy)

    def _run_process(self, work_path):
        pending_requests = []
        ts = str(int(time.time()))
        common_moderation_results_path = self.args.moderation_results_path

        # При добавлении нового партнера с пермарумами не стоит отправлять данные на модерацию в общую таблицу, чтобы не забить очередь модерации
        for name, operator_id, features_schema, beds_schema, moderation_results_path in [
            ('bnovo', hotels_pb2.OI_BNOVO, None, None, common_moderation_results_path),
            ('bronevik', hotels_pb2.OI_BRONEVIK, bronevik_features_schema, bronevik_beds_schema, None),
            ('expedia', hotels_pb2.OI_EXPEDIA, expedia_features_schema, expedia_beds_schema, None),
            ('travelline', hotels_pb2.OI_TRAVELLINE, travelline_features_schema, travelline_beds_schema, common_moderation_results_path)
        ]:
            path = ytlib.join(work_path, name)
            latest_path = ytlib.join(ypath_split(work_path)[0], 'latest', name)
            self.yt_client.create('map_node', path, recursive=True, ignore_existing=True)

            yql_parameters = {
                '$output_path': path,
                '$latest_path': latest_path,
                '$timestamp': ts,
                '$partner_name': name,
                '$operator_id': operator_id,
            }

            if features_schema:
                features_path = ytlib.join(path, 'features-mapping')
                self.yt_client.create("table", features_path, attributes={"schema": features_schema})
                self.yt_client.write_table(features_path, self.feature_reader.read_features(f'{name}_mapping.csv'))
                yql_parameters['$features_mapping_path'] = features_path

            if beds_schema:
                beds_path = ytlib.join(path, 'beds-mapping')
                self.yt_client.create("table", beds_path, attributes={"schema": beds_schema})
                self.yt_client.write_table(beds_path, self.beds_reader.read_beds(f'{name}_beds_mapping.csv'))
                yql_parameters['$beds_mapping_path'] = beds_path

            if moderation_results_path:
                yql_parameters['$moderation_results_path'] = moderation_results_path

            pending_requests.append(yqllib.run_yql_file(
                self.yql_client,
                resource_name=f'{name}.yql',
                project_name='Travel',
                title=f'Build permarooms from {name} feed',
                parameters=yql_parameters,
                sync=False,
            ))

        for request in pending_requests:
            yqllib.wait_results(request)
