# -*- coding: utf-8 -*-

from argparse import ArgumentParser
import logging
import sys
import datetime

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient, ypath_split

from travel.library.python.tools import replace_args_from_env

from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt import ytlib

LOG = logging.getLogger(__name__)

HOURLY_TABLE_NAME = 'hourly'


class Runner:
    def __init__(self, args):
        self.yql_client = YqlClient(db=args.yt_proxy, token=args.yql_token)
        self.yt_client = YtClient(proxy=args.yt_proxy, token=args.yt_token)
        self.args = args

    def run(self):
        if self.args.only_hourly and self.args.only_daily:
            raise Exception("only_hourly and only_daily can't be used simultaneously")
        if not self.args.only_hourly:
            self.build_daily()
        if not self.args.only_daily:
            self.build_houly()

    def build_daily(self):
        if self.args.input_path is None:
            raise Exception("input path is required in non only_hourly mode")
        input_path = self.args.input_path
        input_name = ypath_split(input_path)[1]
        output_path = ytlib.join(self.args.output_dir, input_name)
        min_datetime = None
        if self.args.max_table_age_days is not None:
            min_datetime = datetime.datetime.utcnow() - datetime.timedelta(self.args.max_table_age_days)
        if min_datetime is None or datetime.datetime.strptime(input_name, '%Y-%m-%d') > min_datetime:
            self.run_query(None, input_path, output_path, 0)

    def build_houly(self):
        output_table_names = sorted([x for x in self.yt_client.list(self.args.output_dir) if x != HOURLY_TABLE_NAME])
        last_output_table = output_table_names[-1] if len(output_table_names) > 0 else None
        if self.args.input_path is None or last_output_table == ypath_split(self.args.input_path)[1]:
            if len(output_table_names) == 0:
                min_timestamp = 0
            else:
                min_timestamp = int((datetime.datetime.strptime(output_table_names[-1],
                                                                '%Y-%m-%d') + datetime.timedelta(1)).timestamp())

            output_path = ytlib.join(self.args.output_dir, HOURLY_TABLE_NAME)
            self.run_query(self.args.hourly_input_dir, None, output_path, min_timestamp)

    def run_query(self, input_dir, input_path, output_path, min_timestamp):
        yqllib.run_yql_file(
            self.yql_client,
            resource_name='build_permaroom_logs.yql',
            project_name='Travel',
            title='Build portal permaroom logs',
            parameters={
                '$input_dir': input_dir,
                '$input_path': input_path,
                '$output_path': output_path,
                '$min_timestamp': min_timestamp,
                '$operators_table': self.args.operators_table,
            }
        )


def main():
    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s",
                        stream=sys.stdout)
    logging.getLogger('yt.packages.urllib3.connectionpool').setLevel(logging.WARNING)

    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--yql-token', required=True)
    parser.add_argument('--operators-table', required=True)
    parser.add_argument('--input-path')
    parser.add_argument('--output-dir', required=True)
    parser.add_argument('--hourly-input-dir', required=True)
    parser.add_argument('--only-hourly', action='store_true')
    parser.add_argument('--only-daily', action='store_true')
    parser.add_argument('--max-table-age-days', type=int)
    args = parser.parse_args(args=replace_args_from_env())
    Runner(args).run()


if __name__ == '__main__':
    main()
