from gevent import monkey
monkey.patch_all()

from argparse import ArgumentParser
import gevent
from gevent.subprocess import Popen, PIPE
import logging
import requests
from collections import defaultdict
import sys


READ_DEPTH = 20
PREFIX_WHITELIST = ['/suburban/', '/trains/', '/buses/']


class Replicator(object):
    def __init__(self, replica_url, replica_percent, session_count, stat_owner,):
        self.replica_url = replica_url
        self.replica_percent = replica_percent
        self.stat_owner = stat_owner
        self.free_sessions = []
        self.budget = 0
        for x in range(session_count):
            self.free_sessions.append(requests.session())

    def replicate(self, req):
        self.budget += self.replica_percent
        while self.budget >= 100:
            self.budget -= 100
            if self.free_sessions:
                session = self.free_sessions.pop()
                gevent.spawn(self.replicate_glet, req, session)
                gevent.sleep(0.001)
            else:
                self.stat_owner.stat.on_no_free_session()

    def replicate_glet(self, req, session):
        url = self.replica_url + req
        try:
            session.get(url)
            self.stat_owner.stat.on_ok()
        except Exception as e:
            self.stat_owner.stat.on_fail(e)
        self.free_sessions.append(session)


class Fail(object):
    def __init__(self):
        self.txt = ''
        self.cnt = 0

    def __repr__(self):
        return "%s times, last: '%s'" % (self.cnt, self.txt)


class Stat(object):
    def __init__(self):
        self.ok_cnt = 0
        self.failed = defaultdict(Fail)  # Cause -> count
        self.no_free_session_cnt = 0

    def on_ok(self):
        self.ok_cnt += 1

    def on_fail(self, e):
        f = self.failed[e.__class__]
        f.cnt += 1
        f.txt = str(e)

    def on_no_free_session(self):
        self.no_free_session_cnt += 1


class StatOwner(object):
    def __init__(self):
        self.stat = Stat()
        self.printer = gevent.spawn(self.print_loop)

    def print_loop(self):
        while True:
            gevent.sleep(10)
            stat = self.stat
            self.stat = Stat()
            failed = [(k, v) for k, v in stat.failed.items()]
            failed = sorted(failed, key=lambda p: p[1].cnt, reverse=True)
            logging.info("OK: %s, NoFreeSession: %s, Failed: %s, causes: %s" % (
                stat.ok_cnt, stat.no_free_session_cnt, len(failed), failed))


class LineParseException(Exception):
    pass


class LineIter(object):
    def __init__(self, line):
        self.line = line
        self.pos = 0

    def current(self):
        return self.line[self.pos]

    def advance(self, msg='something'):
        self.pos += 1
        if self.pos > len(self.line):
            raise LineParseException("Unexpected EOL while looking for " + msg)

    def extract(self, start_char, end_char=None):
        if end_char is None:
            end_char = start_char
        while self.current() == ' ':
            self.advance('non-space')
        if start_char != ' ':
            if self.current() != start_char:
                raise LineParseException("Expected '%s' got '%s' at pos %s" % (start_char, self.current(), self.pos))
            self.advance('contents')
        start_pos = self.pos
        while self.current() != end_char:
            self.advance("'%s'" % end_char)
        res = self.line[start_pos:self.pos]
        self.advance('future')
        return res

    def tail(self):
        return self.line[self.pos:]


def read_log(filename, args, kwargs):
    while True:
        error_count = 0
        try:
            logging.info("Opening log file '%s'" % filename)
            tail_args = ['tail', '-F', filename, '-n', str(READ_DEPTH)]
            p = Popen(tail_args, stdout=PIPE, stderr=PIPE, shell=False, close_fds=True)
            while p.returncode is None:
                line = p.stdout.readline()
                try:
                    line = line.decode('utf-8')
                    process_line(line, *args, **kwargs)
                    error_count = 0
                except Exception as e:
                    logging.error("Failed to process log line: %s" % str(e))
                    logging.error("Line: %s" % line)
                    error_count += 1
                    if error_count > 100:
                        raise Exception("Too much errors, restart")
        except Exception as e:
            logging.error("Error during log monitoring: %s" % str(e))


def start_log_mon(filename, *args, **kwargs):
    return gevent.spawn(read_log, filename, args, kwargs)


def process_line(line, replicators):
    line = line.strip()
    if not line:
        return
    iter = LineIter(line)
    level = iter.extract(' ')
    if level != 'INFO:':
        return
    iter.extract(' ')  # date
    iter.extract(' ')  # time
    iter.extract(' ')  # time offset
    iter.extract(' ')  # file:line
    iter.extract(' ')  # Id_xxx:
    action = iter.extract(' ')
    if action != 'Served':
        return
    req = iter.extract("'")
    if not req.startswith('GET /'):
        return
    req = req[4:]
    for pfx in PREFIX_WHITELIST:
        if req.startswith(pfx):
            break
    else:
        return
    for r in replicators:
        r.replicate(req)


def main():
    parser = ArgumentParser()
    parser.add_argument('--log-file', required=True)
    parser.add_argument('--replica-url', action='append')
    parser.add_argument('--replica-percent', required=True, type=int)
    parser.add_argument('--session-count', type=int, default=100)

    args = parser.parse_args()

    logging.basicConfig(level=logging.INFO, format="%(asctime)-15s | %(module)s | %(levelname)s | %(message)s", stream=sys.stdout)

    replicators = []
    for replica_url in args.replica_url:
        logging.info("Replicating to %s" % replica_url)
        r = Replicator(replica_url, args.replica_percent, args.session_count, StatOwner())
        replicators.append(r)
    glet = start_log_mon(args.log_file, replicators)
    glet.join()


if __name__ == '__main__':
    main()
