from abc import ABC, abstractmethod
from dataclasses import dataclass

from yandex_inflector_python import Inflector


INFLECTOR = Inflector()


class IDeclined(ABC):
    @property
    @abstractmethod
    def nominative(self) -> str:
        """Nominative case / именительный падеж"""
        pass

    @property
    @abstractmethod
    def genitive(self) -> str:
        """Genitive case / Родительный падеж"""
        pass

    @property
    @abstractmethod
    def dative(self) -> str:
        """Dative case / дательный падеж"""
        pass

    @property
    @abstractmethod
    def accusative(self) -> str:
        """Accusative case / Винительный падеж"""
        pass

    @property
    @abstractmethod
    def instrumental(self) -> str:
        """Instrumental case / Творительный падеж"""
        pass

    @property
    @abstractmethod
    def prepositional(self) -> str:
        """Prepositional case / Предложный падеж"""
        pass

    @property
    @abstractmethod
    def ablative(self) -> str:
        """Ablative case / Аблатив https://ru.wikipedia.org/wiki/Аблатив"""
        pass

    @property
    @abstractmethod
    def directional(self) -> str:
        """Directional (allative) case / Аллатив https://ru.wikipedia.org/wiki/Аллатив"""
        pass

    @property
    @abstractmethod
    def locative(self) -> str:
        """Locative case / Местный падеж https://ru.wikipedia.org/wiki/Местный_падеж"""
        pass


class DeclinedByInflector(IDeclined):
    def __init__(self, value):
        self.value = value

    @property
    def nominative(self) -> str:
        return self.value  # Do NOT inflect, it is assumed to be nominative initially

    @property
    def genitive(self) -> str:
        return self._inflect("gen")

    @property
    def dative(self) -> str:
        return self._inflect("dat")

    @property
    def accusative(self) -> str:
        return self._inflect("acc")

    @property
    def instrumental(self) -> str:
        return self._inflect("ins")

    @property
    def prepositional(self) -> str:
        return self._inflect("loc")

    @property
    def ablative(self) -> str:
        return "из " + self._inflect("gen")

    @property
    def directional(self) -> str:
        return "в " + self._inflect("acc")

    @property
    def locative(self) -> str:
        return "в " + self._inflect("loc")

    def _inflect(self, grams):
        try:
            inflected = INFLECTOR.Inflect(self.value, grams)
        except Exception as e:
            raise Exception(f"Cannot inflect '{self}' with '{grams}'") from e
        if inflected:
            return inflected
        else:
            return self.value

    def __str__(self):
        return self.nominative


@dataclass
class DeclensionData:
    """
    Данные по склонениям
    """
    nominative: str  # Nominative case / именительный падеж
    genitive: str  # Genitive case / Родительный падеж
    dative: str  # Dative case / дательный падеж
    accusative: str  # Accusative case / Винительный падеж
    instrumental: str  # Instrumental case / Творительный падеж
    prepositional: str  # Prepositional case / Предложный падеж
    preposition: str  # Preposition for prepositional case / Предлог для предложного падежа


class DeclinedByDeclensionData(IDeclined):
    def __init__(self, data: DeclensionData):
        self.data = data

    @property
    def nominative(self) -> str:
        return self.data.nominative

    @property
    def genitive(self) -> str:
        return self.data.genitive

    @property
    def dative(self) -> str:
        return self.data.dative

    @property
    def accusative(self) -> str:
        return self.data.accusative

    @property
    def instrumental(self) -> str:
        return self.data.instrumental

    @property
    def prepositional(self) -> str:
        return f"{self.data.prepositional}"

    @property
    def ablative(self) -> str:
        return f"из {self.data.genitive}"

    @property
    def directional(self) -> str:
        return f"{self.data.preposition} {self.data.accusative}"

    @property
    def locative(self) -> str:
        return f"{self.data.preposition} {self.data.prepositional}"

    def __str__(self):
        return self.nominative
