from pathlib import Path
from typing import Dict, Callable, ClassVar, List, Optional, Type, TypeVar, Union
from dataclasses import dataclass, field
from enum import Enum
import logging

from yaml import dump as dump_yaml, load as load_yaml, Loader, SafeLoader
from marshmallow import Schema
from marshmallow_dataclass import class_schema

from yt.wrapper import YPath, YtClient, write_table
from travel.hotels.lib.python3.tanker_client import TankerClient
from travel.hotels.lib.python3.yt.ytlib import ensure_table_exists, schema_from_dict, link

from travel.hotels.tools.region_pages_builder.common.declined import DeclensionData


FILTER_SLUG_PREFIX = 'filter-'


class IHaveTemplateFields:
    pass


@dataclass
class TankerData:
    pass


@dataclass
class TankerYamlData(TankerData):
    pass


class GeoSearchSortType(Enum):
    RELEVANT_FIRST = "relevant-first"
    CHEAP_FIRST = "cheap-first"
    EXPENSIVE_FIRST = "expensive-first"


@dataclass
class GeoSearchRequestData:
    limit: int = 4
    geoId: Optional[int] = None
    bboxString: Optional[str] = None
    filters: List[str] = field(default_factory=list)
    sortType: Optional[GeoSearchSortType] = field(default=None, metadata={"by_value": True})


@dataclass
class ConfigRegionCrossLinks(TankerYamlData):
    PREFIX: ClassVar[str] = "cross-links"

    class SingleRegionCrossLinksConfig:
        incomingLinksCount: Optional[int] = None
        fixedLinks: List[Union[str, int]] = field(default_factory=list)

    regions: Optional[Dict[str, SingleRegionCrossLinksConfig]] = field(default_factory=dict)


@dataclass
class ConfigAdditionalRegions(TankerYamlData):
    PREFIX: ClassVar[str] = 'additional-regions'

    regionList: List[str]


@dataclass
class ConfigPopularRegions(TankerYamlData):
    PREFIX: ClassVar[str] = 'popular-regions'

    regionList: List[str]


@dataclass
class ConfigRegionFilter(TankerYamlData):
    PREFIX: ClassVar[str] = 'filter.'

    name: str
    filterSlug: str
    category: str
    yqlCondition: str
    regions: List[str]
    geoSearchFilters: List[str] = field(default_factory=list)


@dataclass
class SetFilter:
    slug: str
    name: Optional[str] = None


@dataclass
class FilterSubSet:
    title: str
    filters: List[SetFilter]


@dataclass
class ConfigFilterSet(TankerYamlData):
    PREFIX: ClassVar[str] = 'filter-set'
    template_fields: ClassVar[List[str]] = [
        'title',
    ]

    title: str
    subsets: List[FilterSubSet]


@dataclass
class RegionPage(TankerYamlData):
    PREFIX: ClassVar[str] = 'region.page.'

    seo: str
    content: List[str]
    filterName: Optional[str] = None
    filterSlug: Optional[str] = None


@dataclass
class RegionBlockSeo(TankerYamlData, IHaveTemplateFields):
    PREFIX: ClassVar[str] = 'region.block.seo.'
    template_fields: ClassVar[List[str]] = [
        'title',
        'description',
    ]

    @dataclass
    class IOpenGraphInfo(IHaveTemplateFields):
        template_fields: ClassVar[List[str]] = [
            'title',
            'description',
        ]
        title: str
        description: str

    title: str
    description: str
    openGraph: IOpenGraphInfo


@dataclass
class RegionBlockHeading(TankerYamlData, IHaveTemplateFields):
    PREFIX: ClassVar[str] = 'region.block.search_form.'
    template_fields: ClassVar[List[str]] = [
        'title',
    ]

    title: str
    isCalendarOpen: bool
    geoSearchFilters: List[str] = field(default_factory=list)


@dataclass
class HotelExtraItems:
    nearestStations: bool = False
    cityName: bool = False
    location: bool = False


@dataclass
class RegionBlockHotelList(TankerYamlData, IHaveTemplateFields):
    PREFIX: ClassVar[str] = 'region.block.hotel_list'
    template_fields: ClassVar[List[str]] = [
        'title',
        'buttonText',
    ]

    title: str
    geoSearchRequestData: GeoSearchRequestData
    buttonText: str
    hotelExtraItems: HotelExtraItems = None


@dataclass
class RegionBlockFilterSet(TankerData):
    pass


@dataclass
class MapBlock(TankerData):
    pass


@dataclass
class FiltersBlock(TankerData):
    pass


@dataclass
class BreadcrumbsBlock(TankerData):
    pass


@dataclass
class RegionNameData(TankerYamlData):
    PREFIX: ClassVar[str] = 'region-name'

    data: Dict[str, DeclensionData] = field(default_factory=dict)

    def get_name(self, slug: str) -> DeclensionData:
        return self.data.get(slug)


@dataclass
class RegionCrossLinks(TankerYamlData, IHaveTemplateFields):
    PREFIX: ClassVar[str] = "region.block.cross_links."
    template_fields: ClassVar[List[str]] = [
        'title',
    ]

    title: str


@dataclass
class TankerHtmlData(TankerData):
    content: str


@dataclass
class TextBlock(TankerHtmlData):
    PREFIX: ClassVar[str] = 'region.block.text.'


@dataclass
class FaqBlock(TankerHtmlData):
    PREFIX: ClassVar[str] = 'region.block.faq.'


def parse_tanker_yaml_data(data: str, parse_rule: 'ParseRule') -> TankerData:
    return parse_rule.marshmallow_schema.load(load_yaml(data, Loader=Loader))


def parse_tanker_html_data(data: str, parse_rule: 'ParseRule') -> TankerData:
    return parse_rule.type(data)


@dataclass
class ParseRule:
    prefix: str
    type: Type[TankerData]
    marshmallow_schema: Optional[Schema]
    parse_method: Callable[[str, 'ParseRule'], TankerData]


TankerDataBound = TypeVar('TankerDataBound', bound=TankerData)


class TankerDataStorage:
    PAGE_KEY_ID_PREFIX = RegionPage.PREFIX
    PAGE_KEY_SUFFIX_BY_GEO_ID = 'region-{}'
    PAGE_KEY_SUFFIX_BY_SLUG = 'region-{}'
    PAGE_KEY_SUFFIX_BY_CATEGORY = 'category-{}'
    PAGE_KEY_SUFFIX_BY_REGION_TYPE = 'type-{}'
    PAGE_KEY_SUFFIX_DEFAULT = 'default'

    def __init__(self, raw_tanker_data: Dict[str, str]):
        self.raw_tanker_data = raw_tanker_data
        self.parse_rules = []
        for data_subclass in TankerYamlData.__subclasses__():
            if not hasattr(data_subclass, 'PREFIX'):
                raise Exception(f"PREFIX field missing in class '{data_subclass.__name__}'")
            self.parse_rules.append(ParseRule(
                data_subclass.PREFIX,
                data_subclass,
                class_schema(data_subclass)(),
                parse_tanker_yaml_data
            ))
        self.parse_cache = dict()

        for data_subclass in TankerHtmlData.__subclasses__():
            if not hasattr(data_subclass, 'PREFIX'):
                raise Exception(f"PREFIX field missing in class '{data_subclass.__name__}'")
            self.parse_rules.append(ParseRule(
                data_subclass.PREFIX,
                data_subclass,
                None,
                parse_tanker_html_data
            ))

    def get_config_additional_regions(self) -> ConfigAdditionalRegions:
        raw = self.raw_tanker_data.get(ConfigAdditionalRegions.PREFIX, '{}')
        return self._parse(ConfigAdditionalRegions.PREFIX, raw, ConfigAdditionalRegions)

    def get_config_popular_regions(self) -> ConfigPopularRegions:
        raw = self.raw_tanker_data.get(ConfigPopularRegions.PREFIX, '{}')
        return self._parse(ConfigPopularRegions.PREFIX, raw, ConfigPopularRegions)

    def get_config_region_filters(self) -> Dict[str, ConfigRegionFilter]:
        filters = dict()
        for key, raw in self.raw_tanker_data.items():
            if key.startswith(ConfigRegionFilter.PREFIX):
                filter_item = self._parse(key, raw, ConfigRegionFilter)
                if filter_item.filterSlug.startswith(FILTER_SLUG_PREFIX):
                    filter_item.filterSlug = filter_item.filterSlug[len(FILTER_SLUG_PREFIX):]
                filters[filter_item.filterSlug] = filter_item
        return filters

    def get_config_filter_set(self) -> ConfigFilterSet:
        raw = self.raw_tanker_data.get(ConfigFilterSet.PREFIX, '{}')
        return self._parse(ConfigFilterSet.PREFIX, raw, ConfigFilterSet)

    def get_region_page(
        self,
        geo_id: int,
        slug: str,
        region_type: str,
        filter_slug: Optional[str],
        region_category: Optional[str] = None,
    ) -> RegionPage:
        keys = list()
        if filter_slug is None:
            keys.extend([
                self.PAGE_KEY_ID_PREFIX + self.PAGE_KEY_SUFFIX_BY_SLUG.format(slug),
                self.PAGE_KEY_ID_PREFIX + self.PAGE_KEY_SUFFIX_BY_GEO_ID.format(geo_id),
            ])
        keys.extend([
            self.PAGE_KEY_ID_PREFIX + self.PAGE_KEY_SUFFIX_BY_CATEGORY.format(region_category),
            self.PAGE_KEY_ID_PREFIX + self.PAGE_KEY_SUFFIX_BY_REGION_TYPE.format(region_type),
            self.PAGE_KEY_ID_PREFIX + self.PAGE_KEY_SUFFIX_DEFAULT,
        ])
        for key in keys:
            raw = self.raw_tanker_data.get(key)
            if raw is not None:
                return self._parse(key, raw, RegionPage)
        raise Exception(f"City page template not found for geo_id {geo_id}")

    def get_region_name_data(self) -> RegionNameData:
        raw = self.raw_tanker_data.get(RegionNameData.PREFIX, '{}')
        raw_dict = dict(data=load_yaml(raw, Loader=Loader))
        raw = dump_yaml(raw_dict)
        return self._parse(RegionNameData.PREFIX, raw, RegionNameData)

    def get_cross_links_config(self) -> ConfigRegionCrossLinks:
        return self.get_by_key(ConfigRegionCrossLinks.PREFIX, ConfigRegionCrossLinks)

    def get_by_key(self, key: str, expected_cls: Optional[Type[TankerDataBound]] = None) -> TankerDataBound:
        if key == 'region.block.filter_set':
            return RegionBlockFilterSet()

        if key == 'region.block.filters':
            return FiltersBlock()

        if key == 'region.block.breadcrumbs':
            return BreadcrumbsBlock()

        if key == 'region.block.map':
            return MapBlock()

        raw = self.raw_tanker_data.get(key)
        if raw is None:
            raise Exception(f'Cannot find entry by key {key}')
        try:
            return self._parse(key, raw, expected_cls)
        except Exception as e:
            logging.error(f'Failed to parse {key=}, {raw=}, {expected_cls=}')
            raise e

    def _parse(self, key: str, raw: str, expected_cls: Type[TankerDataBound] = None) -> TankerDataBound:
        v = self.parse_cache.get(key)
        if v is None:
            for rule in self.parse_rules:
                if key.startswith(rule.prefix):
                    v = rule.parse_method(raw, rule)
                    self.parse_cache[key] = v
                    break
            else:
                raise Exception(f'Tanker key {key} has unknown prefix')
        if expected_cls is not None:
            if not isinstance(v, expected_cls):
                raise Exception(f'Key {key}: Expected {expected_cls.__name__} but was {v.__class__.__name__}')
        return v


def read_templates(
    page_name,
    yt_client: YtClient,
    save_state_table: YPath,
    tanker_url: str,
    tanker_project: str,
    templates_keyset: Optional[str],
    templates_yaml_file: Optional[Path] = None,
    templates_yt_table: Optional[YPath] = None,
) -> Optional[TankerDataStorage]:
    source_count = sum(0 if arg is None else 1 for arg in [templates_keyset, templates_yaml_file, templates_yt_table])
    if source_count == 0:
        return
    if source_count > 1:
        raise Exception(f'You should not specify more than one of "keyset", "yaml_file", "yt_table" for {page_name}')

    if templates_yaml_file:
        logging.info(f'Loading templates from file "{templates_yaml_file}"')
        raw_templates = load_yaml(templates_yaml_file.read_text(encoding='utf-8'), Loader=SafeLoader)
    elif templates_yt_table:
        logging.info(f'Loading templates from file "{str(templates_yt_table)}"')
        raw_templates = {}
        for row in yt_client.read_table(templates_yt_table):
            raw_templates[row['key']] = row['value']
    else:
        client = TankerClient(tanker_url)
        raw_templates = client.load_keyset(project_id=tanker_project, keyset_id=templates_keyset)

    ensure_table_exists(save_state_table, yt_client=yt_client, schema=schema_from_dict({
        'key': 'string',
        'value': 'string',
    }))
    table_data = [{'key': k, 'value': v} for k, v in raw_templates.items()]
    write_table(save_state_table, table_data, client=yt_client)

    logging.info(f'{page_name} templates loaded, dumped to {link(save_state_table)}')

    return TankerDataStorage(raw_templates)
