from argparse import ArgumentParser
from collections import Counter
from contextlib import contextmanager
from datetime import date, timedelta
from pathlib import Path
from typing import Any, Callable, Dict, List, NamedTuple, Optional, Set
import json
import logging

from yql.util.type_visitors import print_type
from yt.wrapper import YPath, write_table

from travel.hotels.lib.python3.cli.cli import auto_progress_reporter
from travel.hotels.lib.python3.yt import ytlib
from travel.hotels.lib.python3.yt.ytlib import create_table, ensure_table_exists, schema_from_dict
from travel.hotels.lib.python3.yt.versioned_path import VersionedPath
from travel.hotels.lib.python3.yt.temp_path import TmpPath
from travel.hotels.lib.python3.yql import yqllib
from travel.library.python.tools import replace_args_from_env
from travel.hotels.tools.region_pages_builder.common.tanker_data import (
    ConfigAdditionalRegions, ConfigPopularRegions, ConfigRegionCrossLinks, ConfigRegionFilter, read_templates
)
from travel.hotels.tools.region_pages_builder.data_miner.data_miner.cross_links import (
    CrossLinkGenerator, extract_restrictions_from_cross_links_config
)


FORMAT = '%(asctime)-15s | %(levelname)-4.4s | %(name)-12.12s | %(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)

LINK_COUNT_LIMIT = 12


class RegionKey(NamedTuple):
    geo_id: int
    filter_slug: Optional[str]


class SavedState:

    def __init__(self, state_fn: str, enabled: bool):
        self.state_fn = Path(state_fn)
        self.enabled = enabled
        data = dict(path='', finished_funcs=set())
        if self.state_fn.exists():
            with self.state_fn.open() as f:
                data = json.load(f)
        self.path = data['path']
        self.finished_funcs = set(data['finished_funcs'])

    def exec(self, func: Callable, *args, **kwargs) -> None:
        if not self.enabled:
            func(*args, **kwargs)
            return
        func_name = func.__name__
        if func_name in self.finished_funcs:
            logging.info(f'Skipping {func_name}')
        else:
            func(*args, **kwargs)
        self._save_state(func_name)

    def _save_state(self, func_name: str) -> None:
        self.finished_funcs.add(func_name)
        data = dict(path=str(self.path), finished_funcs=list(self.finished_funcs))
        with self.state_fn.open('w') as f:
            json.dump(data, f)


class Runner:
    NAME = 'RegionPagesBuilderDataMiner'
    LANG = 'ru'

    RUSSIA_GEO_ID = 213

    REGION_LOST_DAYS_LIMIT = timedelta(days=30)

    def __init__(self):
        parser = ArgumentParser()
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token')
        parser.add_argument('--yt-token-path')
        parser.add_argument('--yql-token')
        parser.add_argument('--yql-token-path')
        parser.add_argument('--yt-output-path', default=ytlib.get_default_user_path('region_pages/data_miner'))
        parser.add_argument('--tanker-url', default='https://tanker-api.yandex-team.ru')
        parser.add_argument('--tanker-project', default='travel-backend')
        parser.add_argument('--config-keyset', default=None)
        parser.add_argument('--config-yaml-file', default=None, type=Path)
        parser.add_argument('--yt-region-slugs-path', default='//home/travel/prod/general/slugs/latest')
        parser.add_argument('--yt-popularities-path', default='//home/travel/prod/general/region_pages/popularities')
        parser.add_argument('--yt-accumulated-regions-path', required=True)
        parser.add_argument('--yt-region-prices-path', default='//home/travel/prod/datasets/region_prices/latest')
        parser.add_argument('--yt-region-counters-path', default='//home/travel/prod/datasets/region_counters/latest')
        parser.add_argument('--cluster-permalink-prices', required=True)
        parser.add_argument('--build-additional-regions', action='store_true')
        parser.add_argument('--build-popular-regions', action='store_true')
        parser.add_argument('--calculate-cross-links-top', action='store_true')
        parser.add_argument('--check-region-absence', action='store_true')
        parser.add_argument('--continue-exec', action='store_true', help='Debug only! Resume failed execution')
        args = parser.parse_args(replace_args_from_env())

        self.args = args
        self.yql_client = yqllib.create_client(db=args.yt_proxy, token=args.yql_token, token_path=args.yql_token_path)
        yt_config = {
            'token': args.yt_token,
            'token_path': args.yt_token_path,
        }
        self.yt_client = ytlib.create_client(proxy=args.yt_proxy, config=yt_config)
        self.work_dir = self.args.yt_output_path

        self.saved_state = SavedState('.saved_state.json', self.args.continue_exec)

    def run_yql_file(self, resource_name: str, parameters: Dict[str, Any], template_parameters: Dict[str, Any] = None):
        return yqllib.run_yql_file(
            self.yql_client,
            resource_name,
            self.NAME,
            parameters=parameters,
            template_parameters=template_parameters,
        )

    def run(self):
        with self._get_path() as p, TmpPath(ytlib.join(p, 'tmp'), self.yt_client, keep=True) as tmp, \
             VersionedPath(self.args.yt_accumulated_regions_path, yt_client=self.yt_client) as accumulated_path:
            self.saved_state.path = p

            regions_table = ytlib.join(p, 'regions')
            accumulated_regions_table = ytlib.join(accumulated_path, 'accumulated_regions')
            cities_distance_table = ytlib.join(tmp, 'cities_distance')
            stations_table = ytlib.join(p, 'stations')
            hotel_offers_traits_table = ytlib.join(tmp, 'hotel_offers_traits')
            all_hotels_table = ytlib.join(tmp, 'all_hotels')  # Вообще все отели
            hotels_near_metro_table = ytlib.join(tmp, 'hotels_near_metro')
            hotels_table = ytlib.join(p, 'hotels')  # Те отели, которые интересны рендереру
            city_cross_links_table = ytlib.join(p, 'city_cross_links')
            city_cross_links_link_count_table = ytlib.join(p, 'debug', 'city_cross_metrics')
            popularity_by_albina_table = ytlib.join(tmp, 'popularity-by-albina')
            popularity_by_alinev_table = ytlib.join(tmp, 'popularity-by-alinev')
            prev_path = ytlib.join(self.work_dir, 'latest')

            config = read_templates(
                'cities',
                self.yt_client,
                ytlib.join(p, 'debug', 'tanker-state'),
                self.args.tanker_url,
                self.args.tanker_project,
                self.args.config_keyset,
                self.args.config_yaml_file,
            )

            self.saved_state.exec(self.get_hotel_offers_traits, hotel_offers_traits_table)

            self.saved_state.exec(self.get_all_hotels, hotel_offers_traits_table, all_hotels_table)
            if self.args.check_region_absence:
                self.update_lost_hotels(prev_path, p)

            self.saved_state.exec(self.get_stations, all_hotels_table, stations_table)
            self.saved_state.exec(self.get_hotels_near_metro, all_hotels_table, hotels_near_metro_table)
            self.saved_state.exec(
                self.get_regions, tmp, stations_table, all_hotels_table, hotels_near_metro_table, regions_table,
                config.get_config_additional_regions(), config.get_config_popular_regions(),
                config.get_config_region_filters(), accumulated_regions_table
            )

            if self.args.check_region_absence:
                self.update_lost_regions(prev_path, p)

            self.saved_state.exec(self.get_cities_distance, regions_table, cities_distance_table)

            self.saved_state.exec(
                self.calculate_cross_links,
                config.get_cross_links_config(),
                regions_table,
                cities_distance_table,
                city_cross_links_table,
                city_cross_links_link_count_table,
                popularity_by_albina_table,
                popularity_by_alinev_table,
            )

            self.saved_state.exec(
                self.get_needed_hotels,
                regions_table,
                stations_table,
                all_hotels_table,
                hotels_table
            )

    def get_hotel_offers_traits(self, hotel_offers_traits_table):
        self.run_yql_file(
            'get_hotel_offers_traits.yql',
            parameters={
                '$output_table': hotel_offers_traits_table,
            },
        )

    def get_all_hotels(self, hotel_offers_traits_table, all_hotels_table):
        self.run_yql_file(
            'get_all_hotels.yql',
            parameters={
                '$input_table_hotel_offers_traits': hotel_offers_traits_table,
                '$input_table_hotel_slugs': ytlib.join(self.args.yt_region_slugs_path, 'permalink_to_main_hotel_slug'),
                '$cluster_permalink_prices': self.args.cluster_permalink_prices,
                '$output_table_all_hotels': all_hotels_table,
            },
        )

    def get_stations(self, all_hotels_table, stations_table):
        self.run_yql_file(
            'get_stations.yql',
            parameters={
                '$input_table_rasp_stations': '//home/travel/prod/rasp_dicts/latest/station',
                '$input_table_all_hotels': all_hotels_table,
                '$output_table_stations': stations_table,
            },
        )

    def get_hotels_near_metro(self, all_hotels_table, hotels_near_metro_table):
        self.run_yql_file(
            'get_hotels_near_metro.yql',
            parameters={
                '$input_table_all_hotels': all_hotels_table,
                '$output_table_hotels_near_metro': hotels_near_metro_table,
            },
        )

    def get_regions(
        self,
        temp_path: str,
        stations_table: str,
        all_hotels_table: str,
        hotels_near_metro_table: str,
        regions_table: str,
        config_additional_regions: ConfigAdditionalRegions,
        config_popular_regions: ConfigPopularRegions,
        config_filters: Dict[str, ConfigRegionFilter],
        output_table_accumulated_regions: str,
    ):
        region_schema = schema_from_dict({'slug': 'string', 'filter_slug': 'string'})

        additional_regions_table = ytlib.join(temp_path, 'additional_regions')
        ensure_table_exists(additional_regions_table, yt_client=self.yt_client, schema=region_schema)

        additional_regions_list = config_additional_regions.regionList if self.args.build_additional_regions else list()
        table_data = [{'slug': item} for item in additional_regions_list]
        write_table(additional_regions_table, table_data, client=self.yt_client)

        popular_regions_table = ytlib.join(temp_path, 'popular_regions')
        ensure_table_exists(popular_regions_table, yt_client=self.yt_client, schema=region_schema)

        popular_regions_list = config_popular_regions.regionList if self.args.build_popular_regions else list()
        table_data = [{'slug': item} for item in popular_regions_list]
        write_table(popular_regions_table, table_data, client=self.yt_client)

        input_table_accumulated_regions = ytlib.join(
            self.args.yt_accumulated_regions_path, 'latest', 'accumulated_regions'
        )
        if not ytlib.exists(input_table_accumulated_regions, client=self.yt_client):
            with VersionedPath(self.args.yt_accumulated_regions_path, yt_client=self.yt_client) as p:
                input_table_accumulated_regions = ytlib.join(p, 'accumulated_regions')
            ensure_table_exists(input_table_accumulated_regions, yt_client=self.yt_client, schema=region_schema)

        yql_parameters = {
            '$input_table_region_slugs': ytlib.join(self.args.yt_region_slugs_path, 'region_slugs_export'),
            '$input_table_all_hotels': all_hotels_table,
            '$input_table_stations': stations_table,
            '$input_table_hotels_near_metro': hotels_near_metro_table,
            '$input_table_additional_regions': additional_regions_table,
            '$input_table_popular_regions': popular_regions_table,
            '$input_table_accumulated_regions': input_table_accumulated_regions,
            '$input_table_geoid_prices': ytlib.join(self.args.yt_region_prices_path, 'geoid_prices'),
            '$input_table_geoid_counters': ytlib.join(self.args.yt_region_counters_path, 'geoid_counters'),
            '$output_table_regions': regions_table,
            '$output_table_accumulated_regions': output_table_accumulated_regions,
        }

        template_parameters = list()
        for region_filter in config_filters.values():
            filter_id = region_filter.filterSlug.replace('-', '_')
            table_path = ytlib.join(temp_path, filter_id)
            table_data = [{'slug': item} for item in region_filter.regions]
            ensure_table_exists(table_path, yt_client=self.yt_client, schema=region_schema)
            write_table(table_path, table_data, client=self.yt_client)
            template_parameters.append(dict(
                id=filter_id,
                slug=region_filter.filterSlug,
                category=region_filter.category,
                yql_condition=region_filter.yqlCondition,
            ))
            yql_parameters[f'$input_table_{filter_id}'] = table_path

        self.run_yql_file('get_regions.yql', yql_parameters, dict(filters=template_parameters))

    def get_needed_hotels(self, regions_table, stations_table, all_hotels_table, hotels_table):
        self.run_yql_file(
            'get_needed_hotels.yql',
            parameters={
                '$input_table_regions': regions_table,
                '$input_table_stations': stations_table,
                '$input_table_all_hotels': all_hotels_table,
                '$output_table_hotels': hotels_table,
            },
        )

    def get_cities_distance(self, regions_table: str, cities_distance_table: str):
        yql_parameters = {
            '$input_table_regions': regions_table,
            '$output_table_cities_distance': cities_distance_table,
        }
        self.run_yql_file('get_cities_distance.yql', yql_parameters)

    def get_popularity_by_albina(self, out_table: YPath, regions_table: YPath) -> Callable[[int], float]:
        self.run_yql_file(
            'popularity_by_albina.yql',
            parameters={
                '$in_table': ytlib.join(self.args.yt_popularities_path, 'alyasevenyuk_TRAVELANALYTICS-468_hotels_with_popularity_2019'),
                '$regions_table': regions_table,
                '$out_table': out_table,
            },
        )
        pop_dict = {}
        for row in self._read_yt_table(out_table, 'popularity_by_albina'):
            geo_id = row['geo_id']
            popularity = row['popularity']
            pop_dict[geo_id] = popularity

        def popularity_function(geo_id: int) -> float:
            return pop_dict.get(geo_id, 0)

        return popularity_function

    def get_popularity_by_alinev(self, out_table: YPath) -> Callable[[int], float]:
        self.run_yql_file(
            'popularity_by_alinev.yql',
            parameters={
                '$in_table': ytlib.join(self.args.yt_popularities_path, 'dozornin_marketing_TRAVELWP-155_top-regions'),
                '$out_table': out_table,
            },
        )

        pop_dict = {}
        for row in self._read_yt_table(out_table, 'popularity_by_alinev'):
            geo_id = row['geo_id']
            popularity = row['popularity']
            pop_dict[geo_id] = popularity

        def popularity_function(geo_id: int) -> float:
            return pop_dict.get(geo_id, 0)

        return popularity_function

    def get_popularity_by_hotel_count(self, hotel_count_dict: Dict[int, int]) -> Callable[[int], float]:
        max_hotel_count = float(max(hotel_count_dict.values()))

        def popularity_function(geo_id: int) -> float:
            return hotel_count_dict[geo_id] / max_hotel_count

        return popularity_function

    def calculate_cross_links(
        self,
        cross_links_cfg: ConfigRegionCrossLinks,
        regions_table: YPath,
        cities_distance_table: YPath,
        city_cross_links_table: YPath,
        city_cross_links_link_count_table: YPath,
        popularity_by_albina_table: YPath,
        popularity_by_alinev_table: YPath,
    ):
        cities_by_filter = dict()
        states = set()
        cities_by_state = {}
        slug_to_geo_id = {}
        geo_id_to_name = {}
        geo_id_to_country = {}
        geo_id_to_hotel_count = {}

        for row in self._read_yt_table(regions_table, 'regions'):
            geo_id = row['geo_id']
            name = row['ru_nominative']
            country_geo_id = row['country_id']
            slug = row['slug']
            hotel_count = row['hotel_count']
            filter_slug = row['filter_slug']
            region_type = row['region_type']
            if region_type == 'city' or region_type == 'other':
                filter_cities = cities_by_filter.setdefault(filter_slug, list())
                filter_cities.append(geo_id)
                if filter_slug is None:
                    state_cities = cities_by_state.setdefault(row['state_geo_id'], set())
                    state_cities.add(geo_id)
            elif region_type == 'state':
                states.add(geo_id)
            else:
                raise Exception(f'Unknown region type {region_type}')

            geo_id_to_name[geo_id] = name
            slug_to_geo_id[slug] = geo_id
            geo_id_to_country[geo_id] = country_geo_id
            geo_id_to_hotel_count[geo_id] = hotel_count

        incoming_link_count, fixed_links = extract_restrictions_from_cross_links_config(cross_links_cfg, slug_to_geo_id)

        distance = self.get_distance_function(cities_distance_table)
        popularity_by_albina = self.get_popularity_by_albina(popularity_by_albina_table, regions_table)
        popularity_by_alinev = self.get_popularity_by_alinev(popularity_by_alinev_table)
        popularity_by_hotel_count = self.get_popularity_by_hotel_count(geo_id_to_hotel_count)

        def popularity_function(c_id: int) -> float:
            if geo_id_to_country[c_id] == self.RUSSIA_GEO_ID:
                return 0.5 * popularity_by_hotel_count(c_id) + 0.5 * popularity_by_alinev(c_id) + 0.0 * popularity_by_albina(c_id)
            else:
                return 0.4 * popularity_by_hotel_count(c_id) + 0.1 * popularity_by_alinev(c_id) + 0.5 * popularity_by_albina(c_id)

        def cost_function(c1: int, c2: int) -> float:
            dist = distance(c1, c2)
            popularity_penalty = 3000 * (1 - popularity_function(c2))
            country_mismatch_penalty = 0 if geo_id_to_country[c1] == geo_id_to_country[c2] else 6000
            return dist + popularity_penalty + country_mismatch_penalty

        cross_links = dict()
        for filter_slug, cities in cities_by_filter.items():
            logging.info(f'Calc cross links for {filter_slug}')

            filter_incoming_link_count = incoming_link_count if filter_slug is None else dict()
            filter_fixed_links = fixed_links if filter_slug is None else dict()

            generator = CrossLinkGenerator(
                link_count_limit=LINK_COUNT_LIMIT,
                custom_incoming_link_count=filter_incoming_link_count,
                fixed_links=filter_fixed_links,
                cost_function=cost_function,
                popularity_function=popularity_function,
                city_id_to_name=geo_id_to_name,
            )
            cross_links[filter_slug] = generator.generate_links(cities)

        for filter_slug, graph in cross_links.items():
            energy = 0
            for c_from, links in graph.items():
                energy += sum(map(lambda x: cost_function(c_from, x), links))
            logging.info(f'Energy for {filter_slug}: {energy}')

        self.write_links(city_cross_links_table, cross_links)
        self.write_cross_links_metrics(city_cross_links_link_count_table, cross_links)

        state_to_city_links_path = ytlib.join(self.saved_state.path, 'state_to_city_links')
        state_to_city_links = self.get_state_to_city_links(states, cities_by_state, popularity_function)
        self.write_links(state_to_city_links_path, {None: state_to_city_links})

    def get_distance_function(self, cities_distance_table: YPath) -> Callable[[int, int], float]:
        distance_dict = {}
        for row in self._read_yt_table(cities_distance_table, 'cities_distance'):
            c1_geo_id = row['c1_geo_id']
            c2_geo_id = row['c2_geo_id']
            distance_km = row['distance_km']
            distance_dict[(c1_geo_id, c2_geo_id)] = distance_km
        logging.info(f"Cities distance function loaded, have {len(distance_dict)} distancies")

        def distance(c1: int, c2: int) -> float:
            key = (c1, c2)
            if key not in distance_dict:
                raise ValueError(f"Distance for {c1} -> {c2} not found")
            return distance_dict[key]

        return distance

    @staticmethod
    def get_state_to_city_links(
        states: Set[int],
        cities_by_state: Dict[int, List[int]],
        popularity_function: Callable[[int], float],
    ) -> Dict[int, List[int]]:
        links = dict()
        for state_geo_id, cities in cities_by_state.items():
            if state_geo_id not in states:
                continue
            cities = sorted(cities, key=popularity_function, reverse=True)[:LINK_COUNT_LIMIT]
            links[state_geo_id] = cities
        return links

    def write_links(self, table: YPath, data: Dict[Optional[str], Dict[int, List[int]]]) -> None:
        ensure_table_exists(table, yt_client=self.yt_client, schema=schema_from_dict({
            'filter_slug': 'string',
            'geo_id': 'int32',
            'links': 'any',
        }))
        table_data = [
            {'filter_slug': filter_slug, 'geo_id': geo_id, 'links': links}
            for filter_slug, filter_links in data.items()
            for geo_id, links in filter_links.items()
        ]
        write_table(table, table_data, client=self.yt_client)

    def write_cross_links_metrics(self, table: YPath, data: Dict[Optional[str], Dict[int, List[int]]]) -> None:
        ensure_table_exists(table, yt_client=self.yt_client, schema=schema_from_dict({
            'filter_slug': 'string',
            'geo_id': 'int32',
            'link_count': 'int32',
            'from_cities': 'any',
        }))

        table_data = list()
        for filter_slug, filter_links in data.items():
            table_data.extend(self.get_filter_cross_links_metrics(filter_slug, filter_links))
        write_table(table, table_data, client=self.yt_client)

    @staticmethod
    def get_filter_cross_links_metrics(filter_slug: Optional[str], data: Dict[int, List[int]]) -> List[Dict[str, Any]]:

        counter = Counter()
        who_to_me = {}

        for geo_id, links in data.items():
            counter.update(links)
            for link in links:
                who_to_me.setdefault(link, list()).append(geo_id)

        return [
            {'filter_slug': filter_slug, 'geo_id': geo_id, 'link_count': link_count, 'from_cities': who_to_me[geo_id]}
            for geo_id, link_count in counter.items()
        ]

    def update_lost_regions(self, prev_path: YPath, next_path: YPath) -> None:
        prev_regions_table = ytlib.join(prev_path, 'regions')
        prev_region_last_update_date_table = ytlib.join(prev_path, 'region_last_update_date')
        next_regions_table = ytlib.join(next_path, 'regions')
        next_region_last_update_date_table = ytlib.join(next_path, 'region_last_update_date')

        logging.info(f'Comparing {prev_regions_table} and {next_regions_table}')

        geo_id_to_slug = dict()
        prev_regions = dict()
        for row in self._read_yt_table(prev_regions_table, 'prev'):
            prev_regions[RegionKey(row['geo_id'], row['filter_slug'])] = row
            geo_id_to_slug[row['geo_id']] = row['slug']

        next_regions = dict()
        for row in self._read_yt_table(next_regions_table, 'next'):
            next_regions[RegionKey(row['geo_id'], row['filter_slug'])] = row
            geo_id_to_slug[row['geo_id']] = row['slug']

        prev_keys = set(prev_regions.keys())
        next_keys = set(next_regions.keys())

        new_keys = next_keys - prev_keys
        logging.info(f'{len(new_keys)} new keys: {new_keys}')

        # get prev region_last_update_date
        prev_region_last_update_date = dict()
        if self.yt_client.exists(prev_region_last_update_date_table):
            for row in self.yt_client.read_table(prev_region_last_update_date_table):
                region_key = RegionKey(row['geo_id'], row['filter_slug'])
                prev_region_last_update_date[region_key] = date.fromisoformat(row['last_update_date'])

        lost_keys = []
        for lost_key in prev_keys - next_keys:
            if lost_key in prev_region_last_update_date and date.today() - prev_region_last_update_date[lost_key] > self.REGION_LOST_DAYS_LIMIT:
                logging.error(f'{lost_key} was lost more than {self.REGION_LOST_DAYS_LIMIT.days} days ago, dropping it')
            else:
                lost_keys.append(lost_key)

        # add lost regions
        # implemented using yql https://st.yandex-team.ru/YQL-14893#628cadbe92d5661b9ace5102
        if lost_keys:
            data = (prev_regions[key] for key in lost_keys)
            self.append_to_table(str(next_regions_table), data)

        # build next region_last_update_date
        next_region_last_update_date = dict()
        for region_key in lost_keys:
            next_region_last_update_date[region_key] = prev_region_last_update_date.get(
                region_key,
                date.today() - timedelta(days=1),
            )
        logging.warning(f'{len(next_region_last_update_date)} regions lost: {next_region_last_update_date}')

        # write next region_last_update_date
        schema = schema_from_dict({
            'geo_id': 'uint64',
            'slug': 'string',
            'filter_slug': 'string',
            'last_update_date': 'string',
        })
        create_table(next_region_last_update_date_table, self.yt_client, schema)
        data = (
            {
                'geo_id': key.geo_id,
                'slug': geo_id_to_slug[key.geo_id],
                'filter_slug': key.filter_slug,
                'last_update_date': str(d),
            }
            for key, d in next_region_last_update_date.items()
        )
        self.yt_client.write_table(next_region_last_update_date_table, data)

    def append_to_table(self, table, rows):
        table_iterator = self.yql_client.read_table(table)

        column_names = table_iterator.column_names
        column_types = [print_type(t)[0] for t in table_iterator.column_types]

        self.yql_client.write_table(table, rows, column_names, column_types, append=True)

    def update_lost_hotels(self, prev_path: YPath, next_path: YPath) -> None:
        prev_hotels_table = ytlib.join(prev_path, 'tmp', 'all_hotels')
        next_hotels_table = ytlib.join(next_path, 'tmp', 'all_hotels')
        lost_hotels_table = ytlib.join(next_path, 'tmp', 'lost_hotels')
        yql_parameters = {
            '$input_table_prev_hotels': prev_hotels_table,
            '$output_table_next_hotels': next_hotels_table,
            '$output_table_lost_hotels': lost_hotels_table,
        }
        self.run_yql_file('update_lost_hotels.yql', yql_parameters)

    def _read_yt_table(self, table_path, name):
        for row in auto_progress_reporter(self.yt_client.read_table(table_path), name=name,
                                          total=self.yt_client.row_count(table_path)):
            yield row

    @contextmanager
    def _get_path(self) -> str:
        if self.args.continue_exec and self.saved_state.path:
            yield self.saved_state.path
        else:
            with VersionedPath(self.work_dir, yt_client=self.yt_client) as p:
                yield p
