from typing import Any, Callable, Dict, List, NamedTuple, Optional, Union

from jinja2 import Environment, StrictUndefined

from travel.hotels.tools.region_pages_builder.common.declined import (
    DeclensionData, DeclinedByDeclensionData, DeclinedByInflector, IDeclined
)
from travel.hotels.tools.region_pages_builder.common.tanker_data import ConfigRegionFilter


DECLENSION_FILTERS: Dict[str, Callable[[IDeclined], str]] = {
    'nominative': lambda x: x.nominative,
    'genitive': lambda x: x.genitive,
    'dative': lambda x: x.dative,
    'accusative': lambda x: x.accusative,
    'instrumental': lambda x: x.instrumental,
    'prepositional': lambda x: x.prepositional,
    'ablative': lambda x: x.ablative,
    'directional': lambda x: x.directional,
    'locative': lambda x: x.locative,
}


def link(text: str, link: str) -> str:
    return f'<a href="{link}">{text}</a>'


def rjoin(array: List[Union[Any, IDeclined]], declension: str = None, limit: int = None, joiner: str = ', ', joiner_last: str = ' и ') -> str:
    if declension is None:
        parts = array[:limit]
    else:
        def t(s):
            if isinstance(s, IDeclined):
                pass
            elif isinstance(s, str):
                s = DeclinedByInflector(s)
            else:
                raise Exception(f"Cannot decline object of type {type(s)}: {s}")
            return DECLENSION_FILTERS[declension](s)
        parts = [t(p) for p in array[:limit]]
    if len(parts) == 0:
        return ''
    if len(parts) == 1:
        return parts[0]
    return joiner_last.join([joiner.join(parts[0:len(parts) - 1]), parts[-1]])


FUNCTIONS: List[Callable] = [
    link,
    rjoin,
]


class RegionKey(NamedTuple):
    geo_id: int
    filter_slug: Optional[str]


class Price:
    def __init__(self, value: int, currency: str = "RUB"):
        self.value = value
        self.currency = currency

    def __str__(self):
        return f"<span class=\"price\" currency=\"{self.currency}\">{self.value}</span>"

    def __bool__(self):
        return self.value is not None


class Region(DeclinedByDeclensionData):

    def __init__(self, row: Dict[str, Any], category: Optional[str], lang: str, declension_data: Optional[DeclensionData]):
        if declension_data is None:
            declension_data = DeclensionData(
                nominative=row[f'{lang}_nominative'],
                genitive=row[f'{lang}_genitive'],
                dative=row[f'{lang}_dative'],
                accusative=row[f'{lang}_accusative'],
                instrumental=row[f'{lang}_instrumental'],
                prepositional=row[f'{lang}_prepositional'],
                preposition=row[f'{lang}_preposition'],
            )
        super().__init__(declension_data)

        self.category = category
        self.filter_slug = row.get('filter_slug')

        self.region_type = row['region_type']

        self.geo_id = row['geo_id']
        self.country_geo_id = row['country_id']
        self.slug = row['slug']
        self.hotel_count = row['hotel_count']

        self.top_permalinks = row['top_permalinks']
        self.min_price = Price(row['min_price'])
        self.median_min_price = Price(row['median_min_price'])

        self.top_permalinks_stars_2 = row['top_permalinks_stars_2']
        self.min_price_stars_2 = Price(row['min_price_stars_2'])
        self.median_min_price_stars_2 = Price(row['median_min_price_stars_2'])

        self.top_permalinks_stars_3 = row['top_permalinks_stars_3']
        self.min_price_stars_3 = Price(row['min_price_stars_3'])
        self.median_min_price_stars_3 = Price(row['median_min_price_stars_3'])

        self.top_permalinks_stars_4 = row['top_permalinks_stars_4']
        self.min_price_stars_4 = Price(row['min_price_stars_4'])
        self.median_min_price_stars_4 = Price(row['median_min_price_stars_4'])

        self.top_permalinks_stars_5 = row['top_permalinks_stars_5']
        self.min_price_stars_5 = Price(row['min_price_stars_5'])
        self.median_min_price_stars_5 = Price(row['median_min_price_stars_5'])

        self.top_permalinks_animals_allowed = row['top_permalinks_animals_allowed']
        self.min_price_animals_allowed = Price(row['min_price_animals_allowed'])
        self.median_min_price_animals_allowed = Price(row['median_min_price_animals_allowed'])

        self.top_permalinks_has_spa = row['top_permalinks_has_spa']
        self.min_price_has_spa = Price(row['min_price_has_spa'])
        self.median_min_price_has_spa = Price(row['median_min_price_has_spa'])

        self.top_permalinks_has_bathhouse = row['top_permalinks_has_bathhouse']
        self.min_price_has_bathhouse = Price(row['min_price_has_bathhouse'])
        self.median_min_price_has_bathhouse = Price(row['median_min_price_has_bathhouse'])

        self.top_permalinks_has_sauna = row['top_permalinks_has_sauna']
        self.min_price_has_sauna = Price(row['min_price_has_sauna'])
        self.median_min_price_has_sauna = Price(row['median_min_price_has_sauna'])

        self.top_permalinks_cheap = row['top_permalinks_cheap']
        self.min_price_cheap = Price(row['min_price_cheap'])
        self.median_min_price_cheap = Price(row['median_min_price_cheap'])

        self.top_permalinks_has_pool = row['top_permalinks_has_pool']
        self.min_price_has_pool = Price(row['min_price_has_pool'])
        self.median_min_price_has_pool = Price(row['median_min_price_has_pool'])

        self.top_permalinks_has_private_beach = row['top_permalinks_has_private_beach']
        self.min_price_has_private_beach = Price(row['min_price_has_private_beach'])
        self.median_min_price_has_private_beach = Price(row['median_min_price_has_private_beach'])

        self.top_permalinks_near_metro = row['top_permalinks_near_metro']
        self.min_price_near_metro = Price(row['min_price_near_metro'])
        self.median_min_price_near_metro = Price(row['median_min_price_near_metro'])

        self.top_permalinks_all_included = row['top_permalinks_all_included']
        self.min_price_all_included = Price(row['min_price_all_included'])
        self.median_min_price_all_included = Price(row['median_min_price_all_included'])

        self.top_permalinks_breakfast = row['top_permalinks_breakfast']
        self.min_price_breakfast = Price(row['min_price_breakfast'])
        self.median_min_price_breakfast = Price(row['median_min_price_breakfast'])

        self.top_permalinks_breakfast_lunch_dinner = row['top_permalinks_breakfast_lunch_dinner']
        self.min_price_breakfast_lunch_dinner = Price(row['min_price_breakfast_lunch_dinner'])
        self.median_min_price_breakfast_lunch_dinner = Price(row['median_min_price_breakfast_lunch_dinner'])

        self.top_permalinks_expensive = row['top_permalinks_expensive']
        self.min_price_expensive = Price(row['min_price_expensive'])
        self.median_min_price_expensive = Price(row['median_min_price_expensive'])

        self.airport_station_ids = row['airport_station_ids']
        self.train_station_ids = row['train_station_ids']

        self.has_mir = row.get('has_mir', False)


class Hotel(DeclinedByInflector):
    def __init__(self, row):
        super(Hotel, self).__init__(value=row['name'])
        self.median_min_price = row['median_min_price']
        self.permalink = row['permalink']
        self.slug = row['slug']
        self.stars = row['stars']
        self.rating = row['rating']
        self.top_features = [DeclinedByInflector(f) for f in row['top_features'] or []]  # TODO remove 'or' after dataminer release
        self.price_category = row['price_category']
        self.has_pool = row['has_pool']
        self.has_private_beach = row['has_private_beach']
        self.has_spa = row['has_spa']
        self.animals_allowed = row['animals_allowed']

    @property
    def link(self):
        return Link(self, f'hotel:slug:{self.slug}')

    @property
    def name(self):
        return self


class Station(DeclinedByInflector):
    def __init__(self, row):
        super(Station, self).__init__(value=row['title'])
        self.top_permalinks = row['top_permalinks']
        self.id = row['id']

    @property
    def name(self):
        return self


class Link(IDeclined):
    def __init__(self, wrapped: IDeclined, href: str):
        self.wrapped = wrapped
        self.href = href

    @property
    def nominative(self) -> str:
        return link(self.wrapped.nominative, self.href)

    @property
    def genitive(self) -> str:
        return link(self.wrapped.genitive, self.href)

    @property
    def dative(self) -> str:
        return link(self.wrapped.dative, self.href)

    @property
    def accusative(self) -> str:
        return link(self.wrapped.accusative, self.href)

    @property
    def instrumental(self) -> str:
        return link(self.wrapped.instrumental, self.href)

    @property
    def prepositional(self) -> str:
        return link(self.wrapped.prepositional, self.href)

    @property
    def ablative(self) -> str:
        return link(self.wrapped.ablative, self.href)

    @property
    def directional(self) -> str:
        return link(self.wrapped.directional, self.href)

    @property
    def locative(self) -> str:
        return link(self.wrapped.locative, self.href)

    def __str__(self):
        return self.nominative


class RegionData:

    def __init__(
        self,
        region: Region,
        all_hotels: Dict[int, Hotel],
        stations: Dict[int, Station],
        links: List[Region],
        filter_config: Optional[ConfigRegionFilter],
    ):
        self.region = region
        self.all_hotels = all_hotels
        self.stations = stations
        self.links = links
        self.filter_config = filter_config
        self.dict = self._prepare_dict()

    def _prepare_dict(self) -> Dict:
        result = dict(
            city=self.region,
            state=self.region,
        )
        self._put_hotels(self.region.top_permalinks, 'top_hotels', result)
        self._put_hotels(self.region.top_permalinks_stars_2, 'top_hotels_stars_2', result)
        self._put_hotels(self.region.top_permalinks_stars_3, 'top_hotels_stars_3', result)
        self._put_hotels(self.region.top_permalinks_stars_4, 'top_hotels_stars_4', result)
        self._put_hotels(self.region.top_permalinks_stars_5, 'top_hotels_stars_5', result)
        self._put_hotels(self.region.top_permalinks_cheap, 'top_hotels_cheap', result)
        self._put_hotels(self.region.top_permalinks_has_pool, 'top_hotels_has_pool', result)
        self._put_hotels(self.region.top_permalinks_has_private_beach, 'top_hotels_has_private_beach', result)
        self._put_hotels(self.region.top_permalinks_has_spa, 'top_hotels_has_spa', result)
        self._put_hotels(self.region.top_permalinks_has_bathhouse, 'top_hotels_has_bathhouse', result)
        self._put_hotels(self.region.top_permalinks_has_sauna, 'top_hotels_has_sauna', result)
        self._put_hotels(self.region.top_permalinks_animals_allowed, 'top_hotels_animals_allowed', result)
        self._put_hotels(self.region.top_permalinks_near_metro, 'top_hotels_near_metro', result)
        self._put_hotels(self.region.top_permalinks_all_included, 'top_hotels_all_included', result)
        self._put_hotels(self.region.top_permalinks_expensive, 'top_hotels_expensive', result)
        self._put_hotels(self.region.top_permalinks_breakfast, 'top_hotels_breakfast', result)
        self._put_hotels(self.region.top_permalinks_breakfast_lunch_dinner, 'top_hotels_breakfast_lunch_dinner', result)
        self._put_stations(self.region.airport_station_ids, "airports", result)
        self._put_stations(self.region.train_station_ids, "train_stations", result)
        return result

    def _put_hotels(self, permalinks, name, result):
        idx = 1
        for permalink in permalinks:
            hotel = self.all_hotels[permalink]
            result[f'{name}_{idx}'] = hotel
            idx += 1
        while idx <= 3:
            result[f'{name}_{idx}'] = None
            idx += 1

    def _put_stations(self, station_ids, name, result):
        stations = list()
        for station_id in station_ids:
            if station_id in self.stations:  # precaution, can be not existed station ids on old pages
                station = self.stations[station_id]
                station_info = {
                    'name': station,
                }
                self._put_hotels(station.top_permalinks, "top_hotels", station_info)
                stations.append(station_info)
        result[name] = stations


class Templater:
    def __init__(self):
        self._env = self._prepare_env()

    def render(self, template: str, cities_data: RegionData, on_error_message: Optional[str] = None) -> str:
        try:
            return self._env.from_string(template).render(cities_data.dict)
        except Exception as e:
            if on_error_message:
                raise Exception(on_error_message) from e
            else:
                raise e

    @staticmethod
    def finalize(v):
        if v is None:
            # TODO somehow print template variable name
            raise Exception("Got None during templating")
        return v

    @staticmethod
    def _prepare_env() -> Environment:
        env = Environment(undefined=StrictUndefined, finalize=Templater.finalize)
        for filter_name, filter in DECLENSION_FILTERS.items():
            env.filters[filter_name] = filter
        for function in FUNCTIONS:
            env.globals.update(**{function.__name__: function})
        return env
