from copy import deepcopy
from inspect import isclass
from typing import Dict, Set, TypeVar
import logging

from panamap import Mapper, values_map
from panamap_proto import ProtoMappingDescriptor

from travel.proto.commons_pb2 import TPrice, ECurrency
from travel.hotels.proto.region_pages import region_pages_pb2

from travel.hotels.tools.region_pages_builder.common import tanker_data
from travel.hotels.tools.region_pages_builder.renderer.renderer.exceptions import IHaveExceptionKey
from travel.hotels.tools.region_pages_builder.renderer.renderer import tanker_data_parser as parser
from travel.hotels.tools.region_pages_builder.renderer.renderer.templater import Region, RegionData, Templater


class RenderException(Exception, IHaveExceptionKey):
    def __init__(self, message: str, block_id: str, *key_parts: str):
        super(RenderException, self).__init__(f"Error on rendering block {block_id}: {message}")
        self.key = (block_id, ) + tuple(key_parts)

    def get_key(self):
        return self.key


T = TypeVar('T', bound=tanker_data.TankerYamlData)


class TemplatingMapper:
    def __init__(
        self,
        templater: Templater,
        tanker: tanker_data.TankerDataStorage,
        config: tanker_data.TankerDataStorage,
        available_filters: Dict[int, Set[str]],
    ):
        self.templater = templater
        self.mapper = self._init_mapper()
        self.tanker = tanker
        self.config = config
        self.available_filters = available_filters
        self.html_parser = parser.HtmlParserWrapper()

    def render_page(self, region_data: RegionData, page_type: region_pages_pb2.EPageType) -> region_pages_pb2.TRegionPage:
        result = region_pages_pb2.TRegionPage()
        result.PageType = page_type
        if region_data.filter_config is not None:
            result.FilterName = region_data.filter_config.name
            result.FilterSlug = region_data.filter_config.filterSlug

        page_template: tanker_data.RegionPage = self.render_tanker_yaml_data(
            self.tanker.get_region_page(
                region_data.region.geo_id,
                region_data.region.slug,
                region_data.region.region_type,
                region_data.region.filter_slug,
                region_data.region.category,
            ),
            region_data,
        )

        seo_data: tanker_data.RegionBlockSeo = self.render_tanker_yaml_data(
            self.tanker.get_by_key(page_template.seo, tanker_data.RegionBlockSeo), region_data
        )
        result.SeoBlock.CopyFrom(self.mapper.map(seo_data, region_pages_pb2.TSeoBlock))

        questions_for_seo = []
        for block_id in page_template.content:
            block = self.tanker.get_by_key(block_id)
            block = deepcopy(block)
            if isinstance(block, tanker_data.RegionBlockFilterSet):
                self._add_filter_set(region_data, result)
            elif isinstance(block, tanker_data.BreadcrumbsBlock):
                c = result.Content.add()
                c.BreadcrumbsBlock.CopyFrom(region_pages_pb2.TBreadcrumbsBlock())
            elif isinstance(block, tanker_data.FiltersBlock):
                c = result.Content.add()
                c.FiltersBlock.CopyFrom(region_pages_pb2.TFiltersBlock())
            elif isinstance(block, tanker_data.MapBlock):
                c = result.Content.add()
                c.MapBlock.CopyFrom(region_pages_pb2.TMapBlock())
            elif isinstance(block, tanker_data.TankerYamlData):
                block = self.render_tanker_yaml_data(block, region_data)
                if isinstance(block, tanker_data.RegionBlockHeading):
                    if region_data.filter_config is not None:
                        block.geoSearchFilters = region_data.filter_config.geoSearchFilters
                    c = result.Content.add()
                    c.HeadingBlock.CopyFrom(self.mapper.map(block, region_pages_pb2.THeadingBlock))
                elif isinstance(block, tanker_data.RegionBlockHotelList):
                    if region_data.filter_config is not None:
                        block.geoSearchRequestData.filters.extend(region_data.filter_config.geoSearchFilters)
                    c = result.Content.add()
                    c.HotelListBlock.CopyFrom(self.mapper.map(block, region_pages_pb2.THotelListBlock))
                elif isinstance(block, tanker_data.RegionCrossLinks):
                    self._add_cross_links(region_data, block.title, result)
                else:
                    raise Exception(f'Unknown class {block.__class__}')
            elif isinstance(block, tanker_data.TankerHtmlData):
                raw_html = self.templater.render(
                    block.content,
                    region_data,
                    f"render block '{block_id}' for region_id='{region_data.region.geo_id}'"
                )
                if isinstance(block, tanker_data.TextBlock):
                    try:
                        rendered = self.html_parser.parse(raw_html, parser.TextRenderedBlock)
                    except parser.HtmlDataParsingException as html_e:
                        raise RenderException(f"{html_e.message} on rendering tag\n{html_e.tag}", block_id, html_e.message) from None
                    c = result.Content.add()
                    c.SectionTextBlock.CopyFrom(self.mapper.map(rendered, region_pages_pb2.TSectionTextBlock))
                elif isinstance(block, tanker_data.FaqBlock):
                    try:
                        rendered = self.html_parser.parse(raw_html, parser.FaqRenderedBlock)
                    except parser.HtmlDataParsingException as html_e:
                        raise RenderException(f"{html_e.message} on rendering tag\n{html_e.tag}", block_id, html_e.message) from None
                    if rendered is None:
                        logging.error(f"Cannot render {block_id} for region {region_data.region.nominative}")
                        continue
                    questions_for_seo.extend(rendered.questions)
                    c = result.Content.add()
                    c.SectionTextBlock.CopyFrom(self.mapper.map(rendered, region_pages_pb2.TSectionTextBlock))
                else:
                    raise Exception(f'Unknown class {block.__class__}')
            else:
                raise Exception(f'Unknown class {block.__class__}')

        faq_list = [self.mapper.map(question, region_pages_pb2.TFaqSchemaMarkupItem) for question in questions_for_seo]

        region_schema_org = region_pages_pb2.TGeoRegionSchemaOrgInfo()
        for question in faq_list:
            new_element = region_schema_org.FaqSchemaMarkupItem.add()
            new_element.CopyFrom(question)
        result.SeoBlock.SchemaOrgInfo.CopyFrom(region_pages_pb2.TSchemaOrgInfo(GeoRegionSchemaOrgInfo=region_schema_org))

        return result

    def _add_cross_links(self, region_data: RegionData, title: str, result: region_pages_pb2.TRegionPage):
        link_block = result.Content.add()
        link_block.CopyFrom(region_pages_pb2.TBlock(SectionTextBlock=region_pages_pb2.TSectionTextBlock(
            Title=title,
            Children=[
                region_pages_pb2.TSectionTextBlockContent(
                    GeoLinkGroupBlock=self.mapper.map(city, region_pages_pb2.TGeoLinkGroupBlock)
                )
                for city in region_data.links]
        )))

    def _add_filter_set(self, region_data: RegionData, result: region_pages_pb2.TRegionPage) -> None:
        if region_data.region.filter_slug is not None:
            return
        config_filter_set = self.config.get_config_filter_set()
        config_filter_set = self.render_tanker_yaml_data(config_filter_set, region_data)
        config_region_filters = self.config.get_config_region_filters()

        filter_set = region_pages_pb2.TRegionLinkSetBlock()
        for config_subset in config_filter_set.subsets:
            subset = region_pages_pb2.TRegionLinkSubSetBlock()
            for config_filter in config_subset.filters:
                if config_filter.slug not in self.available_filters[region_data.region.geo_id]:
                    continue
                link = region_pages_pb2.TRegionLinkBlock()
                link.Text = config_filter.name or config_region_filters[config_filter.slug].name
                link.GeoId = region_data.region.geo_id
                link.RegionSlug = region_data.region.slug
                link.FilterSlug = config_filter.slug
                subset.Links.add().CopyFrom(link)
            if len(subset.Links) > 0:
                subset.Title = config_subset.title
                filter_set.Subsets.add().CopyFrom(subset)
        if len(filter_set.Subsets) > 0:
            filter_set.Title = config_filter_set.title
            result.Content.add().CopyFrom(region_pages_pb2.TBlock(RegionLinkSetBlock=filter_set))

    def render_tanker_yaml_data(self, obj: T, city_data: RegionData) -> T:
        template_fields = set(getattr(obj, 'template_fields', []))

        params = {}
        fields = obj.__dataclass_fields__
        for field_name, field_description in fields.items():
            if not self._is_class_var(field_description):
                if field_name in template_fields:
                    params[field_name] = self.templater.render(
                        getattr(obj, field_name),
                        city_data,
                        f"render field '{field_name}' for geo_id='{city_data.region.geo_id}'"
                    )
                elif self._is_list(field_description):
                    params[field_name] = getattr(obj, field_name)  # TODO: Add list expansion
                elif isclass(field_description.type) and issubclass(field_description.type, tanker_data.IHaveTemplateFields):
                    params[field_name] = self.render_tanker_yaml_data(getattr(obj, field_name), city_data)
                else:
                    params[field_name] = getattr(obj, field_name)

        return obj.__class__(**params)

    @staticmethod
    def _is_class_var(field_description: 'Field'):
        return hasattr(field_description.type, '__origin__') and field_description.type.__origin__ is parser.ClassVar

    @staticmethod
    def _is_list(field_description: 'Field'):
        return hasattr(field_description.type, '__origin__') and field_description.type.__origin__ is list

    @staticmethod
    def _init_mapper():
        mapper = Mapper(custom_descriptors=[ProtoMappingDescriptor])

        mapper.mapping(tanker_data.RegionBlockSeo, region_pages_pb2.TSeoBlock) \
            .map_matching(ignore_case=True) \
            .bidirectional("openGraph", "OpenGraphInfo") \
            .register()
        mapper.mapping(tanker_data.RegionBlockSeo.IOpenGraphInfo, region_pages_pb2.TOpenGraphInfo) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(parser.Question, region_pages_pb2.TFaqSchemaMarkupItem) \
            .l_to_r("question", "Question") \
            .l_to_r("answer_raw_text", "Answer") \
            .register()
        mapper.mapping(tanker_data.RegionBlockHeading, region_pages_pb2.THeadingBlock) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(tanker_data.RegionBlockHotelList, region_pages_pb2.THotelListBlock) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(tanker_data.GeoSearchSortType, region_pages_pb2.EGeoSearchSortType) \
            .l_to_r_converter(values_map({
                tanker_data.GeoSearchSortType.RELEVANT_FIRST: region_pages_pb2.EGeoSearchSortType.Value("RELEVANT_FIRST"),
                tanker_data.GeoSearchSortType.CHEAP_FIRST: region_pages_pb2.EGeoSearchSortType.Value("CHEAP_FIRST"),
                tanker_data.GeoSearchSortType.EXPENSIVE_FIRST: region_pages_pb2.EGeoSearchSortType.Value("EXPENSIVE_FIRST"),
            })) \
            .register()
        mapper.mapping(tanker_data.HotelExtraItems, region_pages_pb2.THotelExtraItems) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(tanker_data.GeoSearchRequestData, region_pages_pb2.TGeoSearchRequestData) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(parser.TextRenderedBlock, region_pages_pb2.TSectionTextBlock) \
            .map_matching(ignore_case=True) \
            .register()

        mapper.mapping(parser.SubSectionTextBlock, region_pages_pb2.TSectionTextBlockContent) \
            .map_matching(ignore_case=True) \
            .l_to_r_converter(lambda b: region_pages_pb2.TSectionTextBlockContent(SubSectionTextBlock=mapper.map(b, region_pages_pb2.TSubSectionTextBlock))) \
            .register()

        mapper.mapping(parser.SubSectionTextBlock, region_pages_pb2.TSubSectionTextBlock) \
            .map_matching(ignore_case=True) \
            .register()

        mapper.mapping(parser.FaqRenderedBlock, region_pages_pb2.TSectionTextBlock) \
            .map_matching(ignore_case=True) \
            .l_to_r("questions", "Children") \
            .register()
        mapper.mapping(parser.Question, region_pages_pb2.TSectionTextBlockContent) \
            .l_to_r_converter(lambda q: region_pages_pb2.TSectionTextBlockContent(SpoilerTextBlock=mapper.map(q, region_pages_pb2.TSpoilerTextBlock))) \
            .register()
        mapper.mapping(parser.Question, region_pages_pb2.TSpoilerTextBlock) \
            .l_to_r("question", "Title") \
            .l_to_r("answer", "Description") \
            .register()
        mapper.mapping(parser.Paragraph, region_pages_pb2.TSectionTextBlockContent) \
            .l_to_r_converter(lambda p: region_pages_pb2.TSectionTextBlockContent(Paragraph=mapper.map(p, region_pages_pb2.TParagraph))) \
            .register()
        mapper.mapping(parser.Paragraph, region_pages_pb2.TParagraph) \
            .map_matching(ignore_case=True) \
            .register()

        mapper.mapping(parser.ExternalLinkBlock, region_pages_pb2.TParagraphBlock) \
            .l_to_r_converter(lambda b: region_pages_pb2.TParagraphBlock(ExternalLinkBlock=mapper.map(b, region_pages_pb2.TExternalLinkBlock))) \
            .register()
        mapper.mapping(parser.ExternalLinkBlock, region_pages_pb2.TExternalLinkBlock) \
            .map_matching(ignore_case=True) \
            .register()

        mapper.mapping(parser.HotelLinkBlock, region_pages_pb2.TParagraphBlock) \
            .l_to_r_converter(lambda b: region_pages_pb2.TParagraphBlock(HotelLinkBlock=mapper.map(b, region_pages_pb2.THotelLinkBlock))) \
            .register()
        mapper.mapping(parser.HotelLinkBlock, region_pages_pb2.THotelLinkBlock) \
            .map_matching(ignore_case=True) \
            .register()

        mapper.mapping(parser.PriceTextBlock, region_pages_pb2.TParagraphBlock) \
            .l_to_r_converter(lambda b: region_pages_pb2.TParagraphBlock(PriceTextBlock=mapper.map(b, region_pages_pb2.TPriceTextBlock))) \
            .register()
        mapper.mapping(parser.PriceTextBlock, region_pages_pb2.TPriceTextBlock) \
            .map_matching(ignore_case=True) \
            .register()
        mapper.mapping(parser.Price, TPrice) \
            .l_to_r("value", "Amount", lambda v: int(''.join(v.split(".", maxsplit=2)))) \
            .l_to_r("value", "Precision", lambda v: len(v.split(".", maxsplit=2)[1]) if len(v.split(".", maxsplit=2)) == 2 else 0) \
            .l_to_r("currency", "Currency") \
            .register()
        mapper.mapping(parser.Currency, ECurrency) \
            .l_to_r_converter(values_map({parser.Currency.RUB: ECurrency.Value("C_RUB")})) \
            .register()

        mapper.mapping(parser.PlainTextBlock, region_pages_pb2.TParagraphBlock) \
            .l_to_r_converter(lambda b: region_pages_pb2.TParagraphBlock(PlainTextBlock=mapper.map(b, region_pages_pb2.TPlainTextBlock))) \
            .register()
        mapper.mapping(parser.PlainTextBlock, region_pages_pb2.TPlainTextBlock) \
            .l_to_r("text", "Text") \
            .l_to_r("styles", "Style") \
            .register()
        mapper.mapping(parser.PlainTextBlockStyle, region_pages_pb2.EPlainTextBlockStyle) \
            .l_to_r_converter(values_map({
                parser.PlainTextBlockStyle.ITALIC: region_pages_pb2.EPlainTextBlockStyle.ITALIC,
                parser.PlainTextBlockStyle.BOLD: region_pages_pb2.EPlainTextBlockStyle.BOLD,
            })) \
            .register()

        mapper.mapping(Region, region_pages_pb2.TRegionLinkBlock) \
            .l_to_r("nominative", "Text") \
            .l_to_r("geo_id", "GeoId") \
            .l_to_r("slug", "RegionSlug") \
            .l_to_r("filter_slug", "FilterSlug") \
            .register()
        mapper.mapping(Region, region_pages_pb2.TGeoLinkGroupBlock) \
            .l_to_r_converter(lambda city: region_pages_pb2.TGeoLinkGroupBlock(Main=mapper.map(city, region_pages_pb2.TRegionLinkBlock))) \
            .register()

        return mapper
