from datetime import datetime, timedelta, timezone
from yql.client.parameter_value_builder import YqlParameterValueBuilder as ValueBuilder
import logging

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient
import yt.wrapper as yt

TIMEZONE = timezone(timedelta(hours=3))
YESTERDAY = datetime.now(TIMEZONE).date() - timedelta(days=1)


class Processor:
    """
    replaces logic from SendTravelMetricsToABT from https://a.yandex-team.ru/arc/trunk/arcadia/sandbox/projects/avia/send_travel_metrics_to_abt/__init__.py
    the logic of defining date is moved
    """

    def __init__(
        self,
        yt_client: YtClient,
        yql_client: YqlClient,
        query_args: dict[str, str],
        mr_dir: str,
        tag_name: str,
        date_output_path: str = 'True',
        date_format: str = '%Y-%m-%d',
        debug: bool = False,
        cluster: str = 'hahn'
    ):
        self.yt_client = yt_client
        self.query_args = query_args
        self.debug = debug
        self.mr_dir = mr_dir
        self.tag_name = tag_name
        self.cluster = cluster
        self.date_output_path = date_output_path == 'True'
        self.date_format = date_format

    def pre_process(self):
        self.date = YESTERDAY
        self.query_args['$Date'] = ValueBuilder.make_date(YESTERDAY)
        output_path = self.mr_dir
        if self.date_output_path:
            output_path = yt.ypath_join(output_path, self.date.strftime(self.date_format))

        self.query_args['$OutputPath'] = ValueBuilder.make_string(output_path)

    def post_process(self):
        self.create_and_set_tag(self.date)

    def set_tag(self, tag):
        import requests
        logging.info('Publishing.')
        return requests.get('http://rem-bernstein.n.yandex-team.ru/set_tag?tag=' + tag)

    def create_and_set_tag(self, date):
        tag = self.create_tag(date, self.cluster)
        logging.info('Tag: %s', tag)
        if not self.debug:
            self.set_tag(tag)
        else:
            logging.info('Running in debug mode. Not publishing.')

    def create_tag(self, date, cluster):
        return 'cluster={cluster}_{tag_name}_{date}'.format(
            cluster=cluster,
            tag_name=self.tag_name,
            date=date.strftime('%Y%m%d'),
        )
