import smtplib
import mimetypes
from email.message import EmailMessage
from tempfile import NamedTemporaryFile

from openpyxl import Workbook
from openpyxl.styles import Alignment
from openpyxl.utils import get_column_letter
from openpyxl.worksheet.dimensions import ColumnDimension, DimensionHolder

from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient


class Processor:

    __columns__ = [
        'report_date',
        'nights',
        'horizon',
        'name',
        'type',
        'region',
        'city',
        'address',
        'room_number',
        'stars',
        'min_price',
        'median_min_price',
    ]

    def __init__(
        self,
        yt_client: YtClient,
        yql_client: YqlClient,
        query_args: dict[str, str],
        smtp_server: str,
        sender_username: str,
        recipient: str,
        subject: str,
    ):
        self.yt_client = yt_client
        self.results_path = query_args['$output_table_results']
        self.smtp_server = smtp_server
        self.sender_username = sender_username
        self.recipient = recipient
        self.subject = subject

    def pre_process(self):
        pass

    def post_process(self):
        report = self._get_report()
        self._send_message(report)

    def _get_report(self) -> str:
        wb = Workbook()
        ws = wb.active
        ws.title = self.subject
        ws.append(self.__columns__)
        rows = sorted(
            self.yt_client.read_table(self.results_path, enable_read_parallel=True),
            key=lambda x: x['room_number'] or 0,
            reverse=True,
        )
        for row in rows:
            ws.append([row[col] for col in self.__columns__])

        dimension_holder = DimensionHolder(worksheet=ws)
        for col in range(ws.min_column, ws.max_column + 1):
            dimension_holder[get_column_letter(col)] = ColumnDimension(ws, min=col, max=col, width=20)
        ws.column_dimensions = dimension_holder

        for row in ws.iter_rows():
            for cell in row:
                cell.alignment = Alignment(wrap_text=True, vertical='top')

        with NamedTemporaryFile() as tmp:
            wb.save(tmp.name)
            tmp.seek(0)
            stream = tmp.read()
        return stream

    def _send_message(self, report: str) -> None:
        msg = EmailMessage()
        msg['Subject'] = self.subject
        msg['To'] = self.recipient
        msg['From'] = self.sender_username

        filename = 'report.xlsx'
        content_type, encoding = mimetypes.guess_type(filename)
        if content_type is None or encoding is not None:
            content_type = 'application/octet-stream'
        maintype, subtype = content_type.split('/', 1)
        msg.add_attachment(report, maintype=maintype, subtype=subtype, filename=filename)

        with smtplib.SMTP(self.smtp_server) as s:
            s.send_message(msg)
