from __future__ import print_function

import argparse
import datetime
import logging
import pprint
import socket
import sys
import uuid

import grpc
from google.protobuf import text_format

import travel.proto.commons_pb2 as commons_pb2
import travel.hotels.proto2.hotels_pb2 as hotels_pb2
import travel.hotels.proto2.offer_search_service_pb2_grpc as offer_search_service_pb2_grpc


def protobuf_to_string(message):
    return text_format.MessageToString(message, as_utf8=True)


def create_grpc_channel(args):
    endpoint = "%s:%s" % (args.host, args.port)
    logging.info("creating a channel to '%s'", endpoint)
    return grpc.insecure_channel(endpoint)


def create_metadata():
    metadata = {
        "ya-grpc-call-id": str(uuid.uuid1()),
        "ya-grpc-started-at": datetime.datetime.utcnow().isoformat() + "Z",
        "ya-grpc-timeout-msec": str(60000),
        "ya-grpc-fqdn": socket.getfqdn(),
    }
    return metadata


def handle_ping(args):
    stub = offer_search_service_pb2_grpc.OfferSearchServiceV1Stub(create_grpc_channel(args))
    req = hotels_pb2.TPingRpcReq()
    metadata = create_metadata()
    logging.info("-> REQ %s\n%s\n%s\n", req.__class__, protobuf_to_string(req), pprint.pformat(metadata))
    rsp = stub.Ping(req, metadata=metadata.items())
    logging.info("<- RSP %s\n%s", rsp.__class__, protobuf_to_string(rsp))


def handle_s1o(args):
    stub = offer_search_service_pb2_grpc.OfferSearchServiceV1Stub(create_grpc_channel(args))
    subreqs = []
    if not args.where:
        args.where = ["booking:17"]
    for where in args.where:
        partner_id, original_id = where.split(":")
        partner_id = {
            "b": hotels_pb2.PI_BOOKING,
            "booking": hotels_pb2.PI_BOOKING,
            "101h": hotels_pb2.PI_HOTELS101,
            "101hotels": hotels_pb2.PI_HOTELS101,
            "hc": hotels_pb2.PI_HOTELSCOMBINED,
            "hotelscombined": hotels_pb2.PI_HOTELSCOMBINED,
            "o": hotels_pb2.PI_OSTROVOK,
            "ostrovok": hotels_pb2.PI_OSTROVOK,
            "tl": hotels_pb2.PI_TRAVELLINE,
            "travelline": hotels_pb2.PI_TRAVELLINE,
            "e": hotels_pb2.PI_EXPEDIA,
            "expedia": hotels_pb2.PI_EXPEDIA,
            "d": hotels_pb2.PI_DOLPHIN,
            "dolphin": hotels_pb2.PI_DOLPHIN,
            "bn": hotels_pb2.PI_BNOVO,
            "bnovo": hotels_pb2.PI_BNOVO,
            "tv": hotels_pb2.PI_TVIL,
            "tvil": hotels_pb2.PI_TVIL
        }[partner_id]
        hotel_id = hotels_pb2.THotelId(
            PartnerId=partner_id,
            OriginalId=original_id
        )
        req = hotels_pb2.TSearchOffersReq(
            HotelId=hotel_id,
            CheckInDate=args.checkin,
            CheckOutDate=args.checkout,
            Occupancy=args.occupancy,
            Currency=commons_pb2.C_RUB,
            OfferCacheUseCache=args.use_cache,
            OfferCacheUseSearcher=args.use_searcher,
            RequestClass=hotels_pb2.RC_INTERACTIVE if args.interactive else hotels_pb2.RC_BACKGROUND,
            Id=str(uuid.uuid4()),
            Permalink=args.permalink
        )
        subreqs.append(req)
    reqs = hotels_pb2.TSearchOffersRpcReq(
        Subrequest=subreqs,
        Sync=args.sync,
    )
    metadata = create_metadata()
    logging.info("-> REQ %s\n%rsn%s\n", reqs.__class__, protobuf_to_string(reqs), pprint.pformat(metadata))
    rsps = stub.SearchOffers(reqs, metadata=metadata.items())
    # TODO get and print server metadata
    logging.info("<- RSP %s\n%s", rsps.__class__, protobuf_to_string(rsps))


def _next_friday():
    d = datetime.date.today()
    d += datetime.timedelta(days=((4 - d.weekday()) % 7))
    return d.strftime("%Y-%m-%d")


def _next_sunday_after_next_friday():
    d = datetime.date.today()
    d += datetime.timedelta(days=((4 - d.weekday()) % 7))
    d += datetime.timedelta(days=((6 - d.weekday()) % 7))
    return d.strftime("%Y-%m-%d")


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument("--verbose", action="store_true", default=False)
    parser.add_argument("--host", default="localhost")
    parser.add_argument("--port", default=12574)
    subparsers = parser.add_subparsers()

    s1o_parser = subparsers.add_parser("s1o")
    s1o_parser.set_defaults(func=handle_s1o)
    s1o_parser.add_argument("--where", default=[], action="append")
    s1o_parser.add_argument("--checkin", "-i", default=_next_friday())
    s1o_parser.add_argument("--checkout", "-o", default=_next_sunday_after_next_friday())
    s1o_parser.add_argument("--occupancy", "-g", default="2")
    s1o_parser.add_argument("--async",  action='store_false', dest="sync")
    s1o_parser.add_argument("--use-cache",  action='store_true')
    s1o_parser.add_argument("--use-searcher",  action='store_true')
    s1o_parser.add_argument("--interactive",  action='store_true')
    s1o_parser.add_argument("--permalink", "-p", type=int, default=1)

    ping_parser = subparsers.add_parser("ping")
    ping_parser.set_defaults(func=handle_ping)

    args = parser.parse_args()
    logging.basicConfig(level=(logging.DEBUG if args.verbose else logging.INFO),
                        format="%(asctime)-15s | %(levelname)s | %(message)s",
                        stream=sys.stdout)

    args.func(args)


if __name__ == "__main__":
    main()
