import datetime
import urllib
import xml.etree.cElementTree as ET
from typing import Any, Iterator, Dict

import pytz

YtRow = Dict[str, Any]
SEPARATOR = u' \u2014 '
DESCRIPTION_DATETIME_FORMAT = '%Y.%m.%d в %H:%M'
SETTLEMENT_POINT_KEY_FORMAT = 'c{}'


class TrainSmartBannersConverter(object):
    def convert(self, categories: Iterator[YtRow], trains: Iterator[YtRow]) -> str:
        root = ET.Element("trains-feed")
        ET.SubElement(root, "generation-date").text = datetime.datetime.now().strftime("%Y-%m-%dT%H:%M")

        categories_element = ET.SubElement(root, "categories")
        for category_raw in categories:
            category = ET.SubElement(categories_element, "category", id=str(category_raw['category_id']))
            category.text = category_raw['settlement_from'] + SEPARATOR + category_raw['settlement_to']

        offers_element = ET.SubElement(root, "offers")
        for i, train_row in enumerate(trains):
            self.create_offer_element(offers_element, train_row, i)

        return ET.tostring(root, encoding='utf-8', xml_declaration=True)

    @staticmethod
    def get_local_datetime(utc_timestamp, tz_str):
        time = datetime.datetime.fromtimestamp(utc_timestamp, pytz.UTC)
        return time.astimezone(pytz.timezone(tz_str))

    def create_offer_element(self, offers_element: ET.Element, train_row: YtRow, i: int):
        departure = self.get_local_datetime(train_row['departure'], train_row['from_tz'])
        arrival = self.get_local_datetime(train_row['arrival'], train_row['to_tz'])

        offer_element = ET.SubElement(offers_element, "offer", id=str(i))
        name_element = ET.SubElement(offer_element, "name")
        name_element.text = "Поезд " + train_row["display_number"]
        url_element = ET.SubElement(offer_element, "url")
        params = urllib.parse.urlencode({
            'fromId': SETTLEMENT_POINT_KEY_FORMAT.format(train_row['settlement_id_from']),
            'toId': SETTLEMENT_POINT_KEY_FORMAT.format(train_row['settlement_id_to']),
            'number': train_row['number'],
            'provider': train_row['provider'],
            'time': departure.strftime("%H.%M"),
            'when': departure.strftime("%Y-%m-%d"),
        })
        url_element.text = u"https://travel.yandex.ru/trains/order/?" + params
        picture_element = ET.SubElement(offer_element, "picture")
        picture_element.text = train_row["settlement_image_to"]
        price_element = ET.SubElement(offer_element, "price")
        price_element.text = str(train_row["min_price"])
        currency_id_element = ET.SubElement(offer_element, "currencyId")
        currency_id_element.text = "RUR"
        description_element = ET.SubElement(offer_element, "description")
        description_element.text = "Отправление {}, прибытие {}".format(departure.strftime(DESCRIPTION_DATETIME_FORMAT),
                                                                        arrival.strftime(DESCRIPTION_DATETIME_FORMAT))
        category_id_element = ET.SubElement(offer_element, "categoryId")
        category_id_element.text = str(train_row["category_id"])
        return offer_element
