from argparse import ArgumentParser, Namespace
from typing import Callable
import logging

import yaml
from yt.wrapper import YtClient, write_table

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.tanker_client import TankerClient
from travel.hotels.lib.python3.yt import ytlib
from travel.hotels.lib.python3.yt.versioned_path import KeepLastNCleanupStrategy, VersionedPath


DICTIONARIES_SCHEMA = {
    'slug': 'string',
    'nominative': 'string',
    'genitive': 'string',
    'dative': 'string',
    'accusative': 'string',
    'instrumental': 'string',
    'prepositional': 'string',
    'preposition': 'string',
}


CROSSLINKS_SCHEMA = {
    'key': 'string',
    'template': 'string'
}


class Runner:

    def __init__(self, args: Namespace):
        self.args = args

    def run(self, prepare_method: Callable[[dict[str, str]], list[dict[str, str]]]) -> None:
        logging.info(f'Loading data from {self.args.tanker_keyset}')
        keyset_data = self.get_data_from_tanker()

        data = prepare_method(keyset_data)

        self.save_to_yt(data)
        logging.info('All done')

    def get_data_from_tanker(self) -> dict[str, str]:
        tanker_client = TankerClient(self.args.tanker_url)
        keyset_data = tanker_client.load_keyset(self.args.tanker_project, self.args.tanker_keyset)

        return keyset_data

    def save_to_yt(self, data: list[dict[str, str]]) -> None:
        schema = ytlib.schema_from_dict(args.schema)
        yt_client = YtClient(self.args.yt_proxy, self.args.yt_token)
        cleanup_strategy = KeepLastNCleanupStrategy(self.args.result_keep_count)

        if not yt_client.exists(self.args.dump_path):
            yt_client.create('map_node', path=self.args.dump_path, recursive=True)

        with VersionedPath(self.args.dump_path, yt_client=yt_client, cleanup_strategy=cleanup_strategy) as p:
            table_path = ytlib.join(p, args.table_name)
            logging.info(f'Dumping data to {ytlib.link(table_path)}')
            ytlib.ensure_table_exists(table_path, yt_client=yt_client, schema=schema)
            write_table(table_path, data, client=yt_client)

    def prepare_dictionaries_data(self, keyset_data: dict[str, str]) -> list[dict[str, str]]:
        data_yaml = keyset_data[self.args.tanker_key]
        raw_data = yaml.load(data_yaml, Loader=yaml.BaseLoader)

        data = list()
        for slug, fields in raw_data.items():
            fields['slug'] = slug
            data.append(fields)

        return data

    def prepare_crosslinks_data(self, keyset_data: dict[str, str]) -> list[dict[str, str]]:
        data = list()
        for key, template in keyset_data.items():
            data.append({'key': key, 'template': template})

        return data

    def run_dictionaries(self) -> None:
        self.run(prepare_method=self.prepare_dictionaries_data)

    def run_crosslinks(self) -> None:
        self.run(prepare_method=self.prepare_crosslinks_data)


def run_dictionaries(args: Namespace) -> None:
    Runner(args).run_dictionaries()


def run_crosslinks(args: Namespace) -> None:
    Runner(args).run_crosslinks()


if __name__ == '__main__':
    logging.basicConfig(level=logging.INFO)
    parser = ArgumentParser()

    parser.add_argument('--tanker-url', default='https://tanker-api.yandex-team.ru')
    parser.add_argument('--tanker-project', default='travel-backend')
    parser.add_argument('--tanker-keyset', required=True)
    parser.add_argument('--yt-proxy', required=True)
    parser.add_argument('--yt-token', required=True)
    parser.add_argument('--dump-path', required=True)
    parser.add_argument('--result-keep-count', type=int, default=5)

    subparsers = parser.add_subparsers()
    subparsers.required = True

    dictionaries_parser = subparsers.add_parser('dictionaries')
    dictionaries_parser.set_defaults(func=run_dictionaries)
    dictionaries_parser.set_defaults(table_name='region_names')
    dictionaries_parser.set_defaults(schema=DICTIONARIES_SCHEMA)
    dictionaries_parser.add_argument('--tanker-key', required=True)

    crosslinks_parser = subparsers.add_parser('crosslinks')
    crosslinks_parser.set_defaults(func=run_crosslinks)
    crosslinks_parser.set_defaults(table_name='crosslinks_templates')
    crosslinks_parser.set_defaults(schema=CROSSLINKS_SCHEMA)

    args = parser.parse_args(replace_args_from_env())
    args.func(args)
