from argparse import ArgumentParser
from pathlib import Path
import logging

from yt.wrapper import YPath

from travel.library.python.tools import replace_args_from_env
from travel.hotels.lib.python3.yql import yqllib
from travel.hotels.lib.python3.yt import ytlib
from travel.hotels.tools.text_builder.renderer.renderer.features import load_features
from travel.hotels.tools.text_builder.renderer.renderer.processing import TextRenderer
from travel.hotels.tools.text_builder.renderer.renderer.templates import load_templates


FORMAT = '%(asctime)-15s | %(levelname)-4.4s | %(name)-12.12s | %(message)s'
logging.basicConfig(level=logging.INFO, format=FORMAT)
logging.getLogger('transitions').setLevel(logging.WARNING)


class Runner:
    NAME = 'HotelTextsRenderer'
    LANG = 'ru'

    def __init__(self):
        parser = ArgumentParser()
        parser.add_argument('--yt-proxy', default='hahn')
        parser.add_argument('--yt-token')
        parser.add_argument('--yt-token-path')
        parser.add_argument('--yql-token')
        parser.add_argument('--yql-token-path')
        parser.add_argument('--tanker-url', default='https://tanker-api.yandex-team.ru')
        parser.add_argument('--tanker-project', default='travel-backend')
        parser.add_argument('--templates-keyset', default=None)
        parser.add_argument('--templates-yaml-file', default=None, type=Path)
        parser.add_argument('--features-yaml-file', default=None, type=Path)
        parser.add_argument('--features-yt-table', default=None, type=YPath)
        parser.add_argument('--hotel-permalink', default=None)

        args = parser.parse_args(replace_args_from_env())
        self.args = args

        self.yt_client = None
        self.yql_client = None
        if self.args.yt_proxy:
            yt_config = {
                'token': self.args.yt_token,
                'token_path': self.args.yt_token_path,
            }
            self.yt_client = ytlib.create_client(proxy=self.args.yt_proxy, config=yt_config)
            self.yql_client = yqllib.create_client(db=args.yt_proxy, token=args.yql_token,
                                                   token_path=args.yql_token_path)

    def run(self):
        logging.info("Program started")
        templates = load_templates(
            tanker_url=self.args.tanker_url,
            tanker_project=self.args.tanker_project,
            templates_keyset=self.args.templates_keyset,
            templates_yaml_file=self.args.templates_yaml_file,
        )

        features = load_features(
            yt_client=self.yt_client,
            yql_client=self.yql_client,
            features_yt_table=self.args.features_yt_table,
            features_yaml_file=self.args.features_yaml_file,
            hotel_permalink=self.args.hotel_permalink,
        )

        text_randerer = TextRenderer(templates)
        text = text_randerer.render("Default", features[int(self.args.hotel_permalink)])
        print(f'---------- Begin of text for hotel permalink {self.args.hotel_permalink} ----------')
        print(text)
        print(f'---------- End of text for hotel permalink {self.args.hotel_permalink} ------------')
