from collections.abc import Mapping
from pathlib import Path
from typing import Dict, List, Optional, Iterator
import logging

from yaml import load as load_yaml, SafeLoader
from yql.api.v1.client import YqlClient
from yt.wrapper import YtClient, YPath

from travel.hotels.lib.python3.cli.cli import auto_progress_reporter


class Feature:
    def __init__(self, name, value):
        self.Name = name
        self.Value = value

    def __bool__(self):
        # It allows to check if feature exists for fake features
        return self.Name is not None


class HotelFeatures:
    def __getitem__(self, item):
        # All existed featured are stored as attributes
        fake_feature = Feature(None, None)
        setattr(self, item, fake_feature)
        return fake_feature


def create_features(raw_features: Dict[int, Dict]) -> Dict[int, HotelFeatures]:
    features = {}
    for (permalink, raw_hotel_features) in raw_features.items():
        hotel_features = HotelFeatures()
        for (feature_id, feature_info) in raw_hotel_features.items():
            if isinstance(feature_info, Mapping):
                setattr(hotel_features, feature_id, Feature(feature_info['Name'], feature_info['Value']))
            else:
                setattr(hotel_features, feature_id, Feature(feature_id, feature_info))
        features[permalink] = hotel_features
    return features


def read_yt_table(
    yt_client: YtClient,
    table_path: YPath,
    name: str
) -> Iterator:
    for row in auto_progress_reporter(yt_client.read_table(table_path),
                                      name=name,
                                      total=yt_client.row_count(table_path)):
        yield row


def get_hotel_features_yt_record(
    yql_client: YqlClient,
    table_path: YPath,
    hotel_permalink: int
) -> List[Dict[str, object]]:
    request = yql_client.query(f"SELECT permalink, features FROM `{table_path}` WHERE permalink = {hotel_permalink}")
    request.run()
    values_list = request.full_dataframe.values.tolist()
    return [{
        'permalink': values_list[0][0],
        'features': values_list[0][1],
    }]


def load_features(
    yt_client: Optional = None,
    yql_client: Optional[YqlClient] = None,
    features_yt_table: Optional[YPath] = None,
    features_yaml_file: Optional[Path] = None,
    hotel_permalink: Optional[str] = None
) -> Dict[int, HotelFeatures]:  # features for each hotel permalink
    source_count = sum(0 if arg is None else 1 for arg in [features_yt_table, features_yaml_file])
    if source_count == 0:
        return
    if source_count > 1:
        raise Exception('You should not specify more than one of "yt_features_table_path", "features_yaml_file"')

    if features_yaml_file:
        logging.info(f'Loading features from file "{features_yaml_file}"')
        raw_features = load_yaml(features_yaml_file.read_text(encoding='utf-8'), Loader=SafeLoader)

    if features_yt_table:
        logging.info(f'Loading features from yt table "{features_yt_table}"')
        if hotel_permalink:
            rows = get_hotel_features_yt_record(yql_client, features_yt_table, int(hotel_permalink))
        else:
            rows = read_yt_table(yt_client, features_yt_table, 'features')
        raw_features = {}
        for row in rows:
            hotel_features = row['features']
            raw_features[row['permalink']] = hotel_features

    logging.info(f'Features loaded {len(raw_features)}')

    return create_features(raw_features)
