from abc import abstractmethod
from typing import Dict, Optional
import re

from jinja2 import Environment, StrictUndefined

from travel.hotels.tools.text_builder.renderer.renderer.features import HotelFeatures
from travel.hotels.tools.text_builder.renderer.renderer.templates import HotelTemplates


class IUnitRenderer:
    @abstractmethod
    def render(self,
               template_name: str,
               on_error_message: Optional[str] = None) -> str:
        pass


class UnitRenderer(IUnitRenderer):
    def __init__(self,
                 env: Environment,
                 variables: Dict[str, object],
                 templates: HotelTemplates,
                 template_prefix: str,
                 remove_repeated: Optional[str] = None,
                 remove_all: Optional[str] = None):
        self._env = env
        self._variables = variables
        self._templates = templates
        self._template_prefix = template_prefix
        self._remove_repeated = remove_repeated
        self._remove_all = remove_all

    def render(self,
               template_name: str,
               on_error_message: Optional[str] = None) -> str:
        template = self._templates.get_template(self._template_prefix, template_name)
        try:
            rendered = self._env.from_string(template).render(self._variables)
            if self._remove_all:
                rendered = rendered.replace(self._remove_all, '')
            if self._remove_repeated:
                rendered = re.sub(f'{self._remove_repeated}{{2,}}', f'{self._remove_repeated}', rendered)
            return rendered
        except Exception as e:
            if on_error_message:
                raise Exception(on_error_message) from e
            else:
                raise e


class TemplateVariable:
    def __init__(self, renderer: IUnitRenderer):
        self._renderer = renderer

    def __getitem__(self, item):
        # first usage as a dictionary item, then as an attribute
        value = self._renderer.render(item)
        setattr(self, item, value)
        return value


class TextRenderer:
    def __init__(self, templates: HotelTemplates):
        self._templates = templates
        self._env = self._prepare_env()

    def render(self,
               template_name: str,
               features: HotelFeatures,
               on_error_message: Optional[str] = None) -> str:
        template = self._templates.texts[template_name]
        try:
            sentence_renderer = UnitRenderer(
                env=self._env,
                variables={'Feature': features},
                templates=self._templates,
                template_prefix=HotelTemplates.SENTENCE_TEMPLATE_PREFIX,
                remove_repeated=' ',
                remove_all='\n',
            )
            sentence_var = TemplateVariable(sentence_renderer)

            paragraph_renderer = UnitRenderer(
                env=self._env,
                variables={
                    'Feature': features,
                    HotelTemplates.SENTENCE_TEMPLATE_PREFIX: sentence_var,
                },
                templates=self._templates,
                template_prefix=HotelTemplates.PARAGRAPH_TEMPLATE_PREFIX,
                remove_repeated=' ',
                remove_all='\n',
            )
            paragraph_var = TemplateVariable(paragraph_renderer)

            rendered = self._env.from_string(template).render({
                'Feature': features,
                HotelTemplates.SENTENCE_TEMPLATE_PREFIX: sentence_var,
                HotelTemplates.PARAGRAPH_TEMPLATE_PREFIX: paragraph_var,
            })
            rendered = re.sub('\n{3,}', '\n\n', rendered).strip('\n')
            return rendered
        except Exception as e:
            if on_error_message:
                raise Exception(on_error_message) from e
            else:
                raise e

    @staticmethod
    def finalize(v):
        if v is None:
            raise Exception("Got None during templating")
        return v

    @staticmethod
    def _prepare_env() -> Environment:
        env = Environment(undefined=StrictUndefined, finalize=TextRenderer.finalize)
        return env
