from pathlib import Path
from typing import Dict, Optional
import logging

from yaml import load as load_yaml, SafeLoader

from travel.hotels.lib.python3.tanker_client import TankerClient


class HotelTemplates:
    TEXT_TEMPLATE_PREFIX = 'Text'
    PARAGRAPH_TEMPLATE_PREFIX = 'Paragraph'
    SENTENCE_TEMPLATE_PREFIX = 'Sentence'
    NAME_PART_DELIMITER = '.'

    def __init__(self, raw_tanker_data: Dict[str, str]):
        self.raw_tanker_data = raw_tanker_data
        self.texts = {}
        self.sentences = {}
        self.paragraphs = {}

        self._prefix_to_collection = {
            HotelTemplates.TEXT_TEMPLATE_PREFIX: self.texts,
            HotelTemplates.PARAGRAPH_TEMPLATE_PREFIX: self.paragraphs,
            HotelTemplates.SENTENCE_TEMPLATE_PREFIX: self.sentences,
        }

        for (tanker_key, tanker_value) in raw_tanker_data.items():
            key_parts = [p for p in tanker_key.split(HotelTemplates.NAME_PART_DELIMITER) if p]
            if len(key_parts) != 2:
                raise Exception(f'Cannot parse template key: {tanker_key}')
            if key_parts[0] not in self._prefix_to_collection:
                raise Exception(f'Unknown template prefix "{key_parts[0]}" in key: {tanker_key}')
            self._prefix_to_collection[key_parts[0]][key_parts[1]] = tanker_value

    def get_template(self, prefix: str, name: str):
        return self._prefix_to_collection[prefix][name]


def load_templates(
    tanker_url: str,
    tanker_project: str,
    templates_keyset: Optional[str],
    templates_yaml_file: Optional[Path] = None,
) -> Optional[HotelTemplates]:
    source_count = sum(0 if arg is None else 1 for arg in [templates_keyset, templates_yaml_file])
    if source_count == 0:
        return
    if source_count > 1:
        raise Exception('You should not specify more than one of "keyset", "yaml_file"')

    if templates_yaml_file:
        logging.info(f'Loading templates from file "{templates_yaml_file}"')
        raw_templates = load_yaml(templates_yaml_file.read_text(encoding='utf-8'), Loader=SafeLoader)
    else:
        logging.info(f'Loading templates from tanker keyset "{templates_keyset}"')
        client = TankerClient(tanker_url)
        raw_templates = client.load_keyset(project_id=tanker_project, keyset_id=templates_keyset)

    logging.info(f'Templates loaded {len(raw_templates)}')

    return HotelTemplates(raw_templates)
