from argparse import ArgumentParser
from datetime import date, timedelta
import copy
import json
import requests
import time

from yt import wrapper as yt

import travel.hotels.proto2.hotels_pb2 as hotels_pb2

OFFER_CACHE_HOSTS = {
    "prod": "travel-hotels-offercache.yandex.net",
    "testing": "travel-hotels-offercache-test.yandex.net"
}

OFFER_CACHE_PORT = 11001

DEFAULT_NIGHTS = 1
DEFAULT_AGES = "88,88"
MAX_RETRIES = 10
TL_NAME = hotels_pb2.EPartnerId.Name(hotels_pb2.PI_TRAVELLINE)
WHITELIST_PATH_TEMPLATE = "//home/travel/{0}/config/hotels_whitelist"
OUTPUT_PATH_TEMPLATE = "//home/travel/{0}/general/travelline_offer_checks"


class Colors:
    HEADER = '\033[95m'
    OKBLUE = '\033[94m'
    OKGREEN = '\033[92m'
    WARNING = '\033[93m'
    FAIL = '\033[91m'
    ENDC = '\033[0m'
    BOLD = '\033[1m'
    UNDERLINE = '\033[4m'


def call_offer_cache_loop(env, permalink, date):
    resp = call_offer_cache(env, permalink, date, 0)
    i = 1
    while not resp['IsFinished']:
        time.sleep(1)
        resp = call_offer_cache(env, permalink, date, i)
        if i >= MAX_RETRIES:
            print("Unable to complete search in {0} attempts".format(i))
            break
    return resp


def call_offer_cache(env, permalink, date, request_id):
    url = "http://{0}:{1}/read".format(OFFER_CACHE_HOSTS[env], OFFER_CACHE_PORT)
    s_hotel_id = "{0}~f.singl_org".format(permalink)
    params = {
        "SHotelId": s_hotel_id,
        "Date": date.isoformat(),
        "Nights": DEFAULT_NIGHTS,
        "Ages": DEFAULT_AGES,
        "UseSearcher": 1,
        "RequestId": request_id
    }
    resp = requests.get(url, params)
    assert resp.status_code == 200
    return json.loads(resp.content)


def analyze(hotel):
    if hotel is None:
        return None

    has_travelline_offers = False
    has_non_tl_offers = False
    best_price = None
    best_tl_price = None
    best_non_tl_price = None
    best_partner = None

    for price_item in hotel.get("Prices", []):
        partner_id = price_item["PartnerId"]
        price = price_item["Price"]
        if best_price is None or price < best_price:
            best_partner = [hotels_pb2.EPartnerId.Name(partner_id)]
            best_price = price
        elif best_price is not None and price == best_price:
            best_partner.append(hotels_pb2.EPartnerId.Name(partner_id))

        if partner_id == hotels_pb2.PI_TRAVELLINE:
            has_travelline_offers = True
            if best_tl_price is None or price < best_tl_price:
                best_tl_price = price
        else:
            has_non_tl_offers = True
            if best_non_tl_price is None or price < best_non_tl_price:
                best_non_tl_price = price

    return {
        'has_travelline_offers': has_travelline_offers,
        'has_non_tl_offers': has_non_tl_offers,
        'best_price': best_price,
        'best_tl_price': best_tl_price,
        'best_non_tl_price': best_non_tl_price,
        'best_partner': best_partner
    }


def load_whitelist(client, path):
    return [i for i in client.read_table(path) if i["PartnerIdInt"] == hotels_pb2.PI_TRAVELLINE]


def ensure_output_table(client, path):
    if not client.exists(path):
        client.create("table", path, recursive=True, attributes={
            "schema":
                [
                    {"name": "permalink", "type": "uint64"},
                    {"name": "original_id", "type": "string"},
                    {"name": "checkin_date", "type": "string"},
                    {"name": "has_travelline_offers", "type": "boolean"},
                    {"name": "has_non_tl_offers", "type": "boolean"},
                    {"name": "best_price", "type": "uint64"},
                    {"name": "best_tl_price", "type": "uint64"},
                    {"name": "best_non_tl_price", "type": "uint64"},
                    {"name": "best_partner", "type": "any"},
                    {"name": "timestamp", "type": "uint64"}
                ]
        })


def print_hotel(name, result):
    result = copy.copy(result)

    if result["best_partner"] is None:
        result = "No offers!"
        color = Colors.OKBLUE
    else:
        if result['best_partner'] is not None:
            if TL_NAME in result['best_partner']:
                num_others = len(result['best_partner']) - 1
                if num_others > 0:
                    result['best_partner'] = TL_NAME + " and {0} others".format(num_others)
            else:
                num_others = len(result['best_partner']) - 1
                best_partner = result['best_partner'][0]
                if num_others > 0:
                    result['best_partner'] = best_partner + " and {0} others".format(num_others)

        if not result["has_travelline_offers"]:
            color = Colors.FAIL
        elif result["best_non_tl_price"] is not None and result["best_non_tl_price"] < result["best_tl_price"]:
            color = Colors.WARNING
        elif result["best_non_tl_price"] is None or result["best_non_tl_price"] > result["best_tl_price"]:
            color = Colors.OKGREEN
        else:
            color = Colors.ENDC

    print(color + name + " : " + str(result) + Colors.ENDC)


def check_hotel(env, permalink, original_id, date_to_check, name, timestamp):
    resp = call_offer_cache_loop(env, permalink, date_to_check)
    hotel = resp["Hotels"].get(str(permalink))
    analyzed = analyze(hotel)
    print_hotel(name, analyzed)
    analyzed["timestamp"] = int(timestamp)
    analyzed["permalink"] = permalink
    analyzed["original_id"] = original_id
    analyzed["checkin_date"] = date_to_check.isoformat()
    return analyzed


def get_next_saturday():
    day = date.today()
    while day.weekday() != 6:
        day = day + timedelta(days=1)
    return day


def append_to_output(client, path, results):
    table_path = yt.TablePath(path, append=True)
    client.write_table(table_path, results)


def run(args):
    ts = time.time()
    while_list_path = args.whitelist
    if while_list_path is None:
        while_list_path = WHITELIST_PATH_TEMPLATE.format(args.environment)
    output_path = args.output
    if output_path is None:
        output_path = OUTPUT_PATH_TEMPLATE.format(args.environment)

    client = yt.YtClient(proxy=args.yt_proxy, token=args.yt_token)
    ensure_output_table(client, output_path)
    wl = load_whitelist(client, while_list_path)
    all_results = []
    for hotel in wl:
        permalink = hotel["Permalink"]
        original_id = hotel["OriginalId"]
        print(Colors.BOLD + Colors.UNDERLINE + "Hotel: " + original_id + Colors.ENDC)
        results = [check_hotel(args.environment, permalink, original_id, date.today(), "today       ", ts),
                   check_hotel(args.environment, permalink, original_id, get_next_saturday(), "this weekend", ts),
                   check_hotel(args.environment, permalink, original_id, date.today() + timedelta(weeks=1),
                               "in a week   ",
                               ts),
                   check_hotel(args.environment, permalink, original_id, date.today() + timedelta(days=30),
                               "in a month  ",
                               ts)]
        all_results.extend(results)
    append_to_output(client, output_path, all_results)


if __name__ == '__main__':
    parser = ArgumentParser()
    parser.add_argument('--yt-proxy', default='hahn')
    parser.add_argument('--yt-token')
    parser.add_argument('--environment', default='prod', choices=['prod', 'testing'])
    parser.add_argument('--whitelist')
    parser.add_argument('--output')
    args = parser.parse_args()
    run(args)
