# -*- coding: utf-8 -*-

from datetime import date
from functools import wraps
from enum import Enum
import logging
from yt.wrapper import YtClient, JsonFormat, TablePath


logging.basicConfig(format='%(asctime)s\t%(levelname)s\t%(message)s', level=logging.INFO)


def with_retries(func=None, max_attempts=3):
    if not func:
        return lambda f: with_retries(f, max_attempts=max_attempts)

    @wraps(func)
    def wrapper(*args, **kwargs):
        attempts = max_attempts
        while True:
            try:
                return func(*args, **kwargs)
            except:
                attempts -= 1
                if attempts == 0:
                    raise

    return wrapper


class TurboRecord(object):
    def __init__(self, link, html):
        self.link = link
        self.html = html
        self.pub_date = None
        self.author = None
        self.related = None
        self.related_infinity = False

    def to_dict(self):
        return dict(self.__dict__)

    @staticmethod
    def from_dict(d):
        return TurboRecord(**d)

    @staticmethod
    def get_schema():
        return [
            dict(name='link', type='string'),
            dict(name='html', type='string'),
            dict(name='pub_date', type='uint64'),
            dict(name='author', type='string'),
            dict(name='related', type='string'),
            dict(name='related_infinity', type='boolean'),
        ]

    @staticmethod
    def get_attributes():
        return dict(
            schema=TurboRecord.get_schema(),
            turbo_download_images=False,
            turbo_service_analytics=[],
            turbo_service_advertising=[],
            turbo_service_logo={"title": "Яндекс.Путешествия", "type": "host"},
        )


def create_yt_client():
    # with open('../yt_token.txt') as f:
    #     token = f.read().strip()
    # return YtClient(proxy='hahn', config={'token': token})
    return YtClient(proxy='hahn')


def json_format():
    return JsonFormat(attributes=dict(encode_utf8=False))


@with_retries
def upload(cls, table_path, objects):
    logging.info('uploading {} to YT...'.format(table_path))
    yt_client = create_yt_client()
    with yt_client.Transaction():
        if not yt_client.exists(table_path):
            yt_client.create('table', table_path, attributes=cls.get_attributes())
        yt_client.write_table(table_path, [o.to_dict() for o in objects], format=json_format())
