package ru.yandex.travel.hotels.tugc.grpc

import com.google.protobuf.AbstractMessage
import io.grpc.stub.StreamObserver
import io.grpc.Metadata
import io.grpc.Status
import io.grpc.StatusException
import org.slf4j.LoggerFactory
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.dao.ConcurrencyFailureException
import ru.yandex.travel.commons.grpc.ServerUtils
import ru.yandex.travel.commons.proto.EErrorCode
import ru.yandex.travel.commons.proto.EErrorCode.EC_HOTEL_LIMIT_EXCEEDED
import ru.yandex.travel.commons.proto.Error
import ru.yandex.travel.commons.proto.ErrorException
import ru.yandex.travel.commons.proto.ProtoUtils

import ru.yandex.travel.grpc.GrpcService
import ru.yandex.travel.hotels.common.Permalink
import ru.yandex.travel.hotels.cluster_permalinks.ClusterPermalinkDataProvider
import ru.yandex.travel.hotels.proto.tugc_service.*
import ru.yandex.travel.hotels.tugc.repositories.FavoriteRepository
import ru.yandex.travel.credentials.UserCredentials
import ru.yandex.travel.hotels.tugc.entities.Favorite
import ru.yandex.travel.hotels.tugc.TugcProperties

import ru.yandex.travel.commons.grpc.ServerUtils.statusFromError
import java.util.UUID

@GrpcService(authenticateUser = true, authenticateService = true)
@EnableConfigurationProperties(TugcProperties::class)
class FavoriteGrpcService(
    @Autowired private val clusterPermalinkDataProvider: ClusterPermalinkDataProvider,
    @Autowired private val favoriteRepository: FavoriteRepository,
    private val config: TugcProperties
) : FavoriteInterfaceV1Grpc.FavoriteInterfaceV1ImplBase() {

    private val log = LoggerFactory.getLogger(this.javaClass)

    private fun startTimer() = System.currentTimeMillis()
    private fun logTimer(start: Long, methodName: String, partName: String, reqId: String) {
        val end = System.currentTimeMillis() - start

        log.info("Execute time for $methodName method for $partName part - $end; ReqId $reqId")
    }

    override fun getFavoriteHotels(request: TGetFavoriteHotelsReq, responseObserver: StreamObserver<TGetFavoriteHotelsRsp>) = grpcHandlerWrapper(request, responseObserver) { uc, reqId ->
        val start = startTimer()
        val geoId = if (request.hasGeoId()) request.geoId else null

        val startDB = startTimer()
        val favoriteRecords = favoriteRepository.getAllByUser(uc.parsedPassportIdOrNull, uc.yandexUid, geoId)
        logTimer(startDB, "getFavoriteHotels", "DB Query", reqId)

        val startClusterPermalink = startTimer()
        val favorites = favoriteRecords.mapNotNull { favorite ->
                clusterPermalinkDataProvider.findClusterPermalink(Permalink.of(favorite.permalink))
                    ?.let { favorite.copy(permalink = it.asLong()) }
            }
            .distinctBy { it.permalink }
        logTimer(startClusterPermalink, "getFavoriteHotels", "Cluster permalink", reqId)

        val startResponseBuilder = startTimer()
        val rsp = FavoriteRspBuilder.buildGetFavoriteHotelsRsp(favorites)
        logTimer(startResponseBuilder, "getFavoriteHotels", "Response builder", reqId)

        logTimer(start, "getFavoriteHotels", "All", reqId)

        rsp
    }

    override fun addFavoriteHotel(request: TAddFavoriteHotelReq, responseObserver: StreamObserver<TAddFavoriteHotelRsp>) = grpcHandlerWrapper(request, responseObserver) { uc, reqId ->
        val start = startTimer()
        val geoId = request.geoId

        val startFindCP = startTimer()
        val clusterPermalink = clusterPermalinkDataProvider
            .findClusterPermalink(Permalink.of(request.permalink))
            .asLong()
        logTimer(startFindCP, "addFavoriteHotel", "Find cluster permalink", reqId)

        val startDBFind = startTimer()
        val favoriteRecords = favoriteRepository.getAllByUser(uc.parsedPassportIdOrNull, uc.yandexUid)
        logTimer(startDBFind, "addFavoriteHotel", "DB Query find", reqId)

        val startClusterPermalink = startTimer()
        val favorites = favoriteRecords.mapNotNull { favorite ->
                clusterPermalinkDataProvider.findClusterPermalink(Permalink.of(favorite.permalink))
                    ?.let { favorite.copy(permalink = it.asLong()) }
            }
            .distinctBy { it.permalink }
        logTimer(startClusterPermalink, "addFavoriteHotel", "Cluster permalink", reqId)

        if (favorites.size >= config.maxFavoriteHotels!!) {
            throw Error.with(EC_HOTEL_LIMIT_EXCEEDED, "Favorite hotel limit exceeded for user").toEx()
        }

        val startDBCreate = startTimer()
        favoriteRepository.create(Favorite(uc.parsedPassportIdOrNull, uc.yandexUid, clusterPermalink, geoId))
        logTimer(startDBCreate, "addFavoriteHotel", "DB Query create", reqId)

        val startResponseBuilder = startTimer()
        val rsp = FavoriteRspBuilder.buildAddFavoriteHotelRsp()
        logTimer(startResponseBuilder, "addFavoriteHotel", "Response builder", reqId)

        logTimer(start, "addFavoriteHotel", "All", reqId)

        rsp
    }

    override fun removeFavoriteHotels(request: TRemoveFavoriteHotelsReq, responseObserver: StreamObserver<TRemoveFavoriteHotelsRsp>) = grpcHandlerWrapper(request, responseObserver) { uc, reqId ->
        val start = startTimer()
        when {
            request.hasPermalink() -> {
                val startClusterPermalink = startTimer()
                val permalinks = clusterPermalinkDataProvider
                    .findAllPermalinksOfCluster(Permalink.of(request.permalink))
                    .map { it.asLong() }
                logTimer(startClusterPermalink, "removeFavoriteHotels", "Cluster permalink", reqId)

                val startDBPermalink = startTimer()
                favoriteRepository.deleteByPermalinks(permalinks, uc.parsedPassportIdOrNull, uc.yandexUid)
                logTimer(startDBPermalink, "removeFavoriteHotels", "DB Query delete by permalink", reqId)
            }
            request.hasGeoId() -> {
                val startDBGeoId = startTimer()
                favoriteRepository.deleteByGeoId(request.geoId, uc.parsedPassportIdOrNull, uc.yandexUid)
                logTimer(startDBGeoId, "removeFavoriteHotels", "DB Query delete by geo id", reqId)
            }
            else -> {
                val startDBAll = startTimer()
                favoriteRepository.deleteAll(uc.parsedPassportIdOrNull, uc.yandexUid)
                logTimer(startDBAll, "removeFavoriteHotels", "DB Query delete all", reqId)
            }
        }

        val startResponseBuilder = startTimer()
        val rsp = FavoriteRspBuilder.buildRemoveFavoriteHotelsRsp()
        logTimer(startResponseBuilder, "removeFavoriteHotels", "Response builder", reqId)
        logTimer(start, "removeFavoriteHotels", "All", reqId)

        rsp
    }

    override fun getGeoIds(request: TGetGeoIdsReq, responseObserver: StreamObserver<TGetGeoIdsRsp>) = grpcHandlerWrapper(request, responseObserver) { uc, reqId ->
        val start = startTimer()
        val startDB = startTimer()
        val geoIdsWithCountRecords = favoriteRepository.getAllByUser(uc.parsedPassportIdOrNull, uc.yandexUid)
        logTimer(startDB, "getGeoIds", "DB Query", reqId)

        val startClusterPermalink = startTimer()
        val geoIdsWithCounts = geoIdsWithCountRecords.mapNotNull { favorite ->
                clusterPermalinkDataProvider.findClusterPermalink(Permalink.of(favorite.permalink))
                    ?.let { x -> Pair(favorite, x) }
            }
            .distinctBy { pair -> pair.second }
            .map { pair -> pair.first.geoId }
            .groupingBy { it }
            .eachCount()
        logTimer(startClusterPermalink, "getGeoIds", "Cluster permalink", reqId)

        val startResponseBuilder = startTimer()
        val rsp = FavoriteRspBuilder.buildGetGeoIds(geoIdsWithCounts)
        logTimer(startResponseBuilder, "getGeoIds", "Response builder", reqId)
        logTimer(start, "getGeoIds", "All", reqId)

        rsp
    }

    override fun getHotelFavoriteInfos(request: TGetHotelFavoriteInfosReq, responseObserver: StreamObserver<TGetHotelFavoriteInfosRsp>) = grpcHandlerWrapper(request, responseObserver) { uc, reqId ->
        val start = startTimer()
        val startClusterPermalink = startTimer()
        val permalinksByCluster = request.permalinksList
            .map { permalink -> clusterPermalinkDataProvider.findAllPermalinksOfCluster(Permalink.of(permalink)).map { it.asLong() } }
        logTimer(startClusterPermalink, "getHotelFavoriteInfos", "Cluster permalink", reqId)

        val startDB = startTimer()
        val favorites = favoriteRepository.getAllByPermalinksAndUser(permalinksByCluster.flatten(), uc.parsedPassportIdOrNull, uc.yandexUid)
        logTimer(startDB, "getHotelFavoriteInfos", "DB Query", reqId)

        val startResponseBuilder = startTimer()
        val rsp = FavoriteRspBuilder.buildGetHotelFavoriteInfosRsp(favorites, request.permalinksList, permalinksByCluster)
        logTimer(startResponseBuilder, "getHotelFavoriteInfos", "Response builder", reqId)
        logTimer(start, "getHotelFavoriteInfos", "All", reqId)

        rsp
    }

    private fun <ReqT: AbstractMessage, RspT: AbstractMessage> grpcHandlerWrapper(request: ReqT, responseObserver: StreamObserver<RspT>, block: (uc: UserCredentials, reqId: String) -> RspT) {
        ServerUtils.synchronously(log, request, responseObserver, {
            val uc = UserCredentials.get() ?: throw Error.with(EErrorCode.EC_PERMISSION_DENIED, "credentials are not enabled").toEx()
            val reqId = UUID.randomUUID().toString()

            block(uc, reqId)
        }, { ex -> handleException(request, ex) })
    }

    private fun <ReqT: AbstractMessage> handleException(request: ReqT, ex: Throwable): StatusException {
        log.error("Caught exception ${ex::class.simpleName} while handling request ${request::class.simpleName}", ex)
        var error = ProtoUtils.errorFromThrowable(ex, false)

        if (ex is ConcurrencyFailureException) {
            error = error.toBuilder().setCode(EErrorCode.EC_ABORTED).build()
        }

        val status = if (error.code == EC_HOTEL_LIMIT_EXCEEDED) {
            Status.Code.FAILED_PRECONDITION.toStatus()
                .withDescription(error.message)
                .withCause(ErrorException(error))
        } else {
            statusFromError(error)
        }
        val trailers = Metadata()

        trailers.put(ServerUtils.METADATA_ERROR_KEY, error)

        return status.asException(trailers)
    }
}
