package ru.yandex.travel.hotels.tugc.repositories

import org.jooq.Condition
import org.jooq.DSLContext
import org.jooq.Record

import ru.yandex.travel.hotels.tugc.entities.Favorite
import ru.yandex.travel.hotels.tugc.entities.record
import ru.yandex.travel.hotels.tugc.entities.toEntity
import ru.yandex.travel.hotels.tugc.entities.toFavoriteEntity
import ru.yandex.travel.hotels.tugc.tables.Favorite.FAVORITE
import ru.yandex.travel.hotels.tugc.tables.records.FavoriteRecord

class FavoriteRepository(
    private val dsl: DSLContext
) {
    private fun puid(puid: Long) = "puid_$puid"
    private fun yuid(yuid: String) = "yuid_$yuid"
    private fun userFilter(puid: Long?, yuid: String) = puid?.let {
        FAVORITE.USER_ID.`in`(puid(puid), yuid(yuid))
    } ?: FAVORITE.USER_ID.eq(yuid(yuid))

    fun getAllByUser(puid: Long?, yuid: String, geoId: Int? = null): List<Favorite> {
        var where = userFilter(puid, yuid)

        if (geoId != null) {
            where = where.and(FAVORITE.GEO_ID.eq(geoId))
        }

        val nested = dsl.select()
            .distinctOn(FAVORITE.PERMALINK)
            .from(FAVORITE)
            .where(where)

        return dsl.select()
            .from(nested)
            .orderBy(nested.field(FAVORITE.CREATED_AT).desc())
            .fetch()
            .map(Record::toFavoriteEntity)
    }

    fun getAllByPermalinksAndUser(permalinks: List<Long>, puid: Long?, yuid: String): List<Favorite> {
        val where = userFilter(puid, yuid).and(FAVORITE.PERMALINK.`in`(permalinks))

        return dsl.selectFrom(FAVORITE)
            .where(where)
            .fetch()
            .map(FavoriteRecord::toEntity)
    }

    fun create(favorite: Favorite) {
        val record = favorite.record()

        dsl.insertInto(FAVORITE)
            .set(record)
            .onConflictDoNothing()
            .execute()
    }

    fun deleteByPermalinks(permalinks: List<Long>, puid: Long?, yuid: String) {
        deleteByUser(puid, yuid, FAVORITE.PERMALINK.`in`(permalinks))
    }

    fun deleteByGeoId(geoId: Int, puid: Long?, yuid: String) {
        deleteByUser(puid, yuid, FAVORITE.GEO_ID.eq(geoId))
    }

    fun deleteAll(puid: Long?, yuid: String) {
        deleteByUser(puid, yuid)
    }

    private fun deleteByUser(puid: Long?, yuid: String, filter: Condition? = null) {
        var where = userFilter(puid, yuid)

        if (filter != null) {
            where = where.and(filter)
        }

        dsl.deleteFrom(FAVORITE)
            .where(where)
            .execute()
    }
}
