package main

import (
	"context"
	"fmt"
	stdlog "log"
	"net/http"
	"strconv"

	"github.com/jonboulle/clockwork"
	"golang.yandex/hasql"
	"google.golang.org/grpc"
	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/httputil/middleware/httpmetrics"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	testinggrpc "a.yandex-team.ru/travel/komod/trips/api/testing/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/common/dynamicresources"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api"
	tripsapi "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/async"
	tripsapigrpc "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/grpc"
	tripsapihttp "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/http"
	apiweather "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/weather"
	"a.yandex-team.ru/travel/komod/trips/internal/db"
	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	httputils "a.yandex-team.ru/travel/komod/trips/internal/helpers/http"
	"a.yandex-team.ru/travel/komod/trips/internal/notifier"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/clients"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/mappers"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	activityclients "a.yandex-team.ru/travel/komod/trips/internal/services/activities/clients"
	"a.yandex-team.ru/travel/komod/trips/internal/services/cityimages"
	"a.yandex-team.ru/travel/komod/trips/internal/services/contentadmin"
	"a.yandex-team.ru/travel/komod/trips/internal/services/testing"
	"a.yandex-team.ru/travel/komod/trips/internal/services/unprocessedorders"
	"a.yandex-team.ru/travel/komod/trips/internal/services/weather"
	"a.yandex-team.ru/travel/komod/trips/internal/sharedflights"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	"a.yandex-team.ru/travel/komod/trips/internal/trips/matcher"
	"a.yandex-team.ru/travel/komod/trips/internal/usercredentials"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	"a.yandex-team.ru/travel/library/go/grpcutil/service/servingstatus"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logbroker"
	multilogbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/syncutil"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "travel-trips-api"

func main() {
	maxprocs.AdjustAuto()

	// setting up infrastructure
	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &api.Cfg)
	if err != nil {
		stdlog.Fatalf("can not load configuration: %s", err)
	}

	logger, err := logging.NewDeploy(&api.Cfg.Logging)
	if err != nil {
		stdlog.Fatalf("failed to create logger, err: %s", err)
	}
	defer func() {
		err = logger.L.Sync()
		if err != nil {
			stdlog.Println("failed to close logger:", err)
		}
	}()

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Fatal("tracer close error:", log.Error(err))
		}
	}()

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(api.Cfg.Tvm.Whitelist)
	if api.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient(tvmtool.WithSrc(strconv.Itoa(int(api.Cfg.Tvm.SelfAppID))))
		if err != nil {
			logger.Fatal("failed to create tvm client", log.Error(err))
		}
		tvmClient, err = cachedtvm.NewClient(
			tvmClient,
			cachedtvm.WithCheckServiceTicket(
				api.Cfg.Tvm.CacheTTL,
				api.Cfg.Tvm.CacheMaxItems,
			),
		)
		if err != nil {
			logger.Fatal("failed to create cached tvm client", log.Error(err))
		}
	}

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	referencesRegistry, err := references.NewRegistry(api.Cfg.Dicts)
	if err != nil {
		logger.Fatal("failed to create references registry", log.Error(err))
	}
	settlementByStationExtractor := extractors.NewSettlementByStationExtractor(referencesRegistry)

	healthServer := ghealth.NewServer()
	healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
	runDynamicResourcesService(logger, referencesRegistry)
	servingStatusService := servingstatus.NewService(
		logger,
		serviceName,
		healthServer.SetServingStatus,
		api.Cfg.HealthCheck.UpdateInterval,
		clockwork.NewRealClock(),
	)
	servingStatusService.MonitorServingStatus(ctx)

	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID:    api.Cfg.Tvm.SelfAppID,
			WhiteList: api.Cfg.Tvm.Whitelist,
		},
	)

	geoBase, err := geobase.NewGeobase(api.Cfg.Geobase, logger)
	if err != nil {
		logger.Fatal("failed to create geobase", log.Error(err))
	}
	defer geoBase.Destroy()

	realClock := clockwork.NewRealClock()
	stationIDToSettlementIDMapper := extractors.NewStationIDToSettlementIDMapper(referencesRegistry)
	cachedLocationRepository := helpers.NewCachedLocationRepository()
	pointFactory := point.NewFactory(
		stationIDToSettlementIDMapper,
		cachedLocationRepository,
		geoBase,
		referencesRegistry,
	)

	pgClient, err := pgclient.NewClientBuilder(
		api.Cfg.Database.Hosts,
		api.Cfg.Database.Port,
		api.Cfg.Database.Name,
		api.Cfg.Database.User,
		api.Cfg.Database.Password,
	).WithClusterOptions(
		hasql.WithUpdateTimeout(api.Cfg.Database.HostsUpdateTimeout),
		hasql.WithTracer(
			hasql.Tracer{
				NodeDead: pgclient.OnDeadNode(logger),
			},
		),
	).Build()
	if err != nil {
		logger.Fatal("failed to build pgClient", log.Error(err))
	}

	tripsService := db.NewTripsStorage(db.DefaultTransactionOptions, pgClient, pointFactory)
	ordersHTTPClient := clients.NewHTTPClient(
		api.Cfg.OrdersClient,
		&http.Client{Timeout: api.Cfg.OrdersClient.RequestTimeout},
		tvmClient,
	)
	ordersClient := clients.NewMappingClient(
		logger,
		ordersHTTPClient,
		mappers.NewAviaOrderMapper(
			referencesRegistry,
			cachedLocationRepository,
			settlementByStationExtractor,
		),
		mappers.NewHotelOrderMapper(geoBase, referencesRegistry, pointFactory),
		mappers.NewTrainOrderMapper(
			referencesRegistry,
			cachedLocationRepository,
			settlementByStationExtractor,
		),
		mappers.NewBusOrderMapper(referencesRegistry, cachedLocationRepository, settlementByStationExtractor),
	)

	var unprocessedOrdersLogbrokerProducer = createUnprocessedOrderLogbrokerProducer(ctx, logger)
	defer unprocessedOrdersLogbrokerProducer.Close()
	unprocessedOrdersStorage := db.NewUnprocessedOrdersStorage(pgClient)
	var unprocessedOrdersService = unprocessedorders.NewService(
		logger,
		unprocessedOrdersLogbrokerProducer,
		realClock,
		unprocessedOrdersStorage,
	)

	pointResolver := point.NewResolver(geoBase, referencesRegistry, pointFactory)
	pointComparator := point.NewComparator(geoBase, referencesRegistry, pointResolver)
	spanComparator := span.NewSpanComparator(pointComparator)
	spansHelper := span.NewHelper(pointComparator, spanComparator)
	contentAdminClient, err := contentadmin.NewRetryableClient(logger, api.Cfg.Tvm.SelfAppID, api.Cfg.ContentAdmin)
	if err != nil {
		logger.Fatal("failed to build content admin retryable client", log.Error(err))
	}
	cityImagesService := cityimages.NewService(logger, api.Cfg.CityImages, contentAdminClient)
	go cityImagesService.RunCaching(ctx)
	pointImagesExtractor := extractors.NewPointImagesExtractor(logger, cityImagesService, referencesRegistry)

	afishaClient := activityclients.NewAfishaHTTPClient(
		api.Cfg.AfishaClient,
		httputils.NewHTTPFetcher(
			api.Cfg.AfishaClient.TravelAPITvmID,
			&http.Client{Timeout: api.Cfg.AfishaClient.RequestTimeout},
			tvmClient,
			httputils.WithRequestTimeout(api.Cfg.AfishaClient.RequestTimeout),
		),
	)

	iziTravelClient := activityclients.NewIziTravelHTTPClient(
		api.Cfg.IziTravelClient,
		httputils.NewHTTPFetcher(
			api.Cfg.IziTravelClient.TravelAPITvmID,
			&http.Client{Timeout: api.Cfg.IziTravelClient.RequestTimeout},
			tvmClient,
			httputils.WithRequestTimeout(api.Cfg.IziTravelClient.RequestTimeout),
		),
	)

	activitiesProvider := async.NewActivitiesProvider(logger, spansHelper, afishaClient, iziTravelClient, pointResolver)
	notifierGRPCConn, err := notifier.CreateConnection(api.Cfg.Tvm.SelfAppID, api.Cfg.Notifier, logger)
	if err != nil {
		logger.Fatal("failed to establish connection to travel-notifier", log.Error(err))
	}
	notifierClient := notifier.NewNotificationsClient(notifierGRPCConn, api.Cfg.Notifier, logger)
	sharedFlightsClient := sharedflights.NewClient(api.Cfg.SharedFlights, &http.Client{}, logger)

	weatherClient := weather.NewHTTPClient(
		api.Cfg.WeatherClient,
		httputils.NewHTTPFetcher(
			api.Cfg.WeatherClient.TravelAPITvmID,
			&http.Client{Timeout: api.Cfg.WeatherClient.RequestTimeout},
			tvmClient,
			httputils.WithRequestTimeout(api.Cfg.WeatherClient.RequestTimeout),
		),
	)
	weatherRequestExtractor := apiweather.NewRequestExtractor(spansHelper, pointComparator, realClock)
	weatherProvider := apiweather.NewProvider(
		api.Cfg.WeatherProvider,
		weatherClient,
		logger,
		weatherRequestExtractor,
	)

	tripsProvider := tripsapi.NewTripsProvider(
		api.Cfg.Provider,
		api.Cfg.RaspMediaURL,
		logger,
		pgClient,
		tripsService,
		ordersClient,
		extractors.NewOrderInfoExtractor(pointFactory),
		tripsapi.NewStartPageBuilder(realClock),
		geoBase,
		tripsapi.NewTripsMapper(
			logger,
			spansHelper,
			api.Cfg.RaspMediaURL,
			pointImagesExtractor,
			tripsapi.NewTrainDescriptionBuilder(logger),
			referencesRegistry.Carriers(),
		),
		async.NewBlockProvider(
			logger,
			activitiesProvider,
		),
		unprocessedOrdersService,
		tripsapi.NewRestrictionsProvider(
			logger,
			spansHelper,
			pointResolver,
			pointFactory,
		),
		tripsapi.NewCrossSaleProvider(
			logger,
			spansHelper,
		),
		notifierClient,
		weatherProvider,
		sharedFlightsClient,
		referencesRegistry,
		stationIDToSettlementIDMapper,
	)

	grpcService := tripsapigrpc.NewService(logger, tripsProvider, unprocessedOrdersService)
	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		grpcService.GetServiceRegisterer(),
	}

	httpHandler := tripsapihttp.NewHandler(logger, tripsProvider)

	go func() {
		err = metricsserver.RunMetricsHTTPServer(
			context.Background(),
			metricsserver.DefaultMetricsConfig,
			logger,
			rootRegistry,
		)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()
	const swaggerPrefix = "/api"
	wg := syncutil.WaitGroup{}
	testingTripsProvider := tripsapi.NewTestingTripsProvider(
		tripsProvider,
		matcher.NewRuleFactory(pointComparator, spansHelper),
		tripsapi.NewStartPageBuilder(realClock),
	)
	testingService := testing.NewService(testingTripsProvider, referencesRegistry, pointFactory)
	wg.Go(func() {
		runTestingAPI(testingService, logger, rootRegistry)
	})

	probesState := buildProbesState(
		ctx,
		logger.WithName("server.probe").(*zap.Logger),
		healthServer,
		tvmHelper,
		pgClient,
	)

	if api.Cfg.Testing.Enabled {
		wg.Go(
			func() {
				grpcGateway := grpcgateway.NewGateway(
					&api.Cfg.Testing.GrpcGateway,
					grpcgateway.NewService(
						"testing",
						swaggerPrefix,
						api.Cfg.Testing.GrpcAddr,
						testinggrpc.RegisterTestingServiceV1HandlerFromEndpoint,
						nil,
					),
				)
				logger.Info("Started Testing API")
				err := grpcGateway.Run(ctx)
				if err != nil {
					logger.Fatal("Error while starting grpc gateway", log.Error(err))
				}
			},
		)
	}
	wg.Go(
		func() {
			var serverTvmClient tvm.Client
			if !api.Cfg.IsDevelopment() {
				serverTvmClient = tvmClient
			}

			server := grpcserver.NewGrpcServerBuilder(
				api.Cfg.Grpc,
				logger.WithName("servers.grpc").(*zap.Logger),
			).
				WithRegisterers(grpcServiceRegisterers...).
				WithHealthServer(healthServer).
				WithInterceptors(
					helpers.MakeMetricsInterceptor(rootRegistry.WithPrefix("grpc")),
					usercredentials.NewUnpackInterceptor(),
				).
				WithOptionalTVM(serverTvmClient, tvmAllowedIds).
				Build()
			logger.Info(
				"Started GRPC",
				log.String("addr", api.Cfg.Grpc.Addr),
			)
			err = server.Run(ctx)
			if err != nil {
				logger.Fatal("Error while starting Grpc server", log.Error(err))
			}
		},
	)

	wg.Go(
		func() {
			var serverTvmClient tvm.Client
			if !api.Cfg.IsDevelopment() {
				serverTvmClient = tvmClient
			}

			routeBuilders := append(
				probes.GetChiRouteBuilders(&api.Cfg.Probes, probesState),
				httpHandler.GetRouteBuilder(),
			)

			server := httpserver.NewHTTPServerBuilder(
				api.Cfg.HTTP,
				logger,
				routeBuilders,
				rootRegistry.WithPrefix("http"),
			).
				WithMetricsMiddlewareOptions(httpmetrics.WithDurationBuckets(metrics.DefaultTimingsBuckets)).
				WithMiddlewares(getMiddlewares(logger, serverTvmClient)...).
				Build()
			logger.Info(
				"Started HTTP",
				log.String("addr", api.Cfg.HTTP.Addr),
			)
			err = server.Run(ctx)
			if err != nil {
				logger.Fatal("Error while starting HTTP server", log.Error(err))
			}
		},
	)
	wg.Wait()
}

func runTestingAPI(service *testing.Service, logger *zap.Logger, rootRegistry coreMetrics.Registry) {
	server := grpcserver.NewGrpcServerBuilder(grpcserver.GrpcConfig{Addr: api.Cfg.Testing.GrpcAddr}, logger).
		WithMetrics(rootRegistry.WithPrefix("testing-grpc")).
		WithRegisterers(service.GetServiceRegisterer()).
		Build()
	err := server.Run(context.Background())
	if err != nil {
		logger.Fatal("Error while starting testing Grpc server", log.Error(err))
	}
}

func createUnprocessedOrderLogbrokerProducer(
	ctx context.Context,
	logger log.Logger,
) unprocessedorders.LogbrokerProducer {
	if api.Cfg.UnprocessedOrders.Mock {
		return unprocessedorders.NewMockProducer()
	}
	logbrokerProducer, err := logbroker.NewProducer(
		api.Cfg.UnprocessedOrders.Topic,
		api.Cfg.UnprocessedOrders.Endpoint,
		multilogbroker.GetDeploySourceID(api.Cfg.UnprocessedOrders.ProducerID),
		logbroker.NewOAuthCredentialsProvider(api.Cfg.UnprocessedOrders.Token),
		logger,
		logbroker.WithoutSeqNo(),
	)
	if err != nil {
		logger.Fatal("failed to create logbroker producer", log.Error(err))
	}
	err = logbrokerProducer.Run(ctx)
	if err != nil {
		logger.Fatal("failed to run logbroker producer", log.Error(err))
	}
	return logbrokerProducer
}

func runDynamicResourcesService(logger log.Logger, dictsRegistry *references.Registry) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		api.Cfg.DynamicResources,
		dynamicresources.WithOnUpdateDicts(dictsRegistry.OnUpdate),
	)
	dynamicResourceService.BackgroundRun()
}

func buildProbesState(ctx context.Context, logger log.Logger, healthServer *ghealth.Server, tvmHelper tvmutil.TvmHelper, pgClient *pgclient.Client) *probes.State {
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(api.Cfg.Tvm.SelfAppID)
	if tvmInterceptor == nil {
		logger.Fatal("failed to create tvm interceptor for readiness server")
	}

	onReady := func() error {
		dialOptions := []grpc.DialOption{
			grpc.WithInsecure(),
			grpc.WithBlock(),
		}
		if !api.Cfg.IsDevelopment() {
			logger.Info("add tvm interceptor to readiness server")
			dialOptions = append(dialOptions, grpc.WithUnaryInterceptor(tvmInterceptor))
		}

		clientConn, err := grpc.DialContext(
			ctx,
			api.Cfg.Grpc.Addr,
			dialOptions...,
		)
		if err != nil {
			return fmt.Errorf("unable to dial connection: %w", err)
		}
		defer clientConn.Close()
		response, err := healthpb.NewHealthClient(clientConn).Check(
			ctx,
			&healthpb.HealthCheckRequest{Service: serviceName},
		)
		if err != nil {
			return fmt.Errorf("failed to check health service: %w", err)
		}
		if response.GetStatus() != healthpb.HealthCheckResponse_SERVING {
			return fmt.Errorf("failed to check health service")
		}
		return nil
	}
	onStop := func() {
		healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_NOT_SERVING)
	}

	return probes.NewState(
		logger,
		probes.OnReady(pgClient.Ping),
		probes.OnReady(onReady),
		probes.OnStopBefore(onStop),
	)
}
