package main

import (
	"context"
	"fmt"
	"io"
	"io/ioutil"
	"os"
	"os/exec"
	"strings"

	"google.golang.org/grpc"
	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

func getContextWithCredentials(ctx context.Context, yandexUID, passportID, userTicket, serviceTicket string) context.Context {
	return metadata.AppendToOutgoingContext(
		ctx,
		"X-Ya-User-Ticket", userTicket,
		"X-Ya-Service-Ticket", serviceTicket,

		"x-ya-user-ticket", userTicket,
		"x-ya-yandexuid", yandexUID,
		"x-ya-passportid", passportID,
	)
}

func getUserTicket(dstTvmID int, tvmSecret, oauthToken string) string {
	oauthFile := createSecretFile("oauth", oauthToken)
	defer os.Remove(oauthFile.Name())

	tvmSecretFile := createSecretFile("tvm-secret", tvmSecret)
	defer os.Remove(tvmSecretFile.Name())

	cmd := exec.Command(
		"ya",
		"tool",
		"tvmknife",
		"get_user_ticket",
		"oauth",
		"-b", Cfg.BlackboxHost,
		fmt.Sprintf("--tvm_id=%d", dstTvmID),
		"-t", oauthFile.Name(),
		"-S", tvmSecretFile.Name(),
	)
	logger.Debug("getting user ticket", log.String("cmd", cmd.String()))

	value, err := cmd.Output()
	if err != nil {
		logger.Fatal(
			"unable to execute tvmknife",
			log.Error(err),
		)
	}

	ticket := strings.TrimSpace(string(value))
	logger.Debug(
		"read user ticket",
		log.String("value", ticket),
	)

	return ticket
}

func getServiceTicket(srcTvmID, dstTvmID int, tvmSecret string) string {
	tvmSecretFile := createSecretFile("tvm-secret", tvmSecret)
	defer os.Remove(tvmSecretFile.Name())

	cmd := exec.Command(
		"ya",
		"tool",
		"tvmknife",
		"get_service_ticket",
		"client_credentials",
		fmt.Sprintf("--src=%d", srcTvmID),
		fmt.Sprintf("--dst=%d", dstTvmID),
		"-S", tvmSecretFile.Name(),
	)
	logger.Debug("getting service ticket", log.String("cmd", cmd.String()))

	value, err := cmd.Output()
	if err != nil {
		logger.Fatal(
			"unable to execute tvmknife",
			log.Error(err),
		)
	}

	ticket := strings.TrimSpace(string(value))
	logger.Debug(
		"read service ticket",
		log.String("value", ticket),
	)

	return ticket
}

func createSecretFile(pattern, value string) *os.File {
	secretFile, err := ioutil.TempFile("/tmp", pattern)
	if err != nil {
		logger.Fatal(
			"unable to create temporary file",
			log.Error(err),
			log.String("pattern", pattern),
		)
	}
	_, err = io.WriteString(secretFile, value)
	if err != nil {
		logger.Fatal(
			"unable to write secret into temporary file",
			log.Error(err),
			log.String("pattern", pattern),
		)
	}
	return secretFile
}

func getConnection() *grpc.ClientConn {
	if Cfg.LocalAddr != "" {
		var opts []grpc.DialOption
		opts = append(opts, grpc.WithInsecure())

		conn, err := grpc.Dial(Cfg.LocalAddr, opts...)
		if err != nil {
			logger.Fatal(
				"fail to dial",
				log.Error(err),
				log.String("local_addr", Cfg.LocalAddr),
			)
		}
		return conn

	}
	if Cfg.YPlannerID != "" {
		conn, err := helpers.CreateConnection(
			Cfg.YPlannerID,
			uint32(Cfg.TvmID),
			logger,
			getTvmHelper(),
		)
		if err != nil {
			logger.Fatal(
				"fail to dial",
				log.Error(err),
				log.String("y_planner_id", Cfg.YPlannerID),
			)
		}
		return conn
	}
	logger.Fatal("should define y_planner_id or local_addr")
	return nil
}

func getTvmHelper() *tvmutil.TvmHelper {
	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID: uint32(Cfg.TvmID),
		},
	)
	return &tvmHelper
}
