package main

import (
	"context"
	"io/ioutil"
	"os"
	"strings"
	"time"

	"golang.yandex/hasql"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/services/unprocessedorders"
	"a.yandex-team.ru/travel/library/go/containers"
)

type ordersCollector struct {
	logger                   log.Logger
	pgClient                 *pgclient.Client
	unprocessedOrdersService *unprocessedorders.Service
	config                   Config
}

func newOrdersCollector(
	logger log.Logger,
	pgClient *pgclient.Client,
	unprocessedOrdersService *unprocessedorders.Service,
	config Config,
) *ordersCollector {
	return &ordersCollector{
		logger:                   logger,
		pgClient:                 pgClient,
		unprocessedOrdersService: unprocessedOrdersService,
		config:                   config,
	}
}

func (oc *ordersCollector) collectAndSend(ctx context.Context) {
	previouslyCollected, err := loadPreviouslyCollectedOrders(oc.config.InputFile)
	if err != nil {
		oc.logger.Fatal(err.Error())
	}
	collectedOrders, err := oc.collectOrders()
	if err != nil {
		oc.logger.Fatal(err.Error())
	}
	extendWithNewCollected(collectedOrders, previouslyCollected)
	if err := dumpOrders(oc.config.OutputFile, previouslyCollected); err != nil {
		oc.logger.Fatal(err.Error())
	}
	if oc.config.SendToLogbroker {
		failedToSendOrders := oc.sendToLogbroker(ctx, previouslyCollected)
		if err := dumpOrders(oc.config.FailedToSendOrdersFile, failedToSendOrders); err != nil {
			oc.logger.Fatal(err.Error())
		}
	}
}

func (oc *ordersCollector) collectOrders() ([]string, error) {
	oc.logger.Info("collecting orders started")
	defer oc.logger.Info("collecting orders finished")
	ctx, cancel := context.WithTimeout(context.Background(), oc.config.GlobalTimeout)
	defer cancel()
	db, err := oc.pgClient.GetDB(hasql.PreferStandby)
	if err != nil {
		return nil, err
	}
	db = db.WithContext(ctx)
	if oc.config.Debug {
		db = db.Debug()
	}
	totalOrders, err := getTotalOrdersCount(db)
	if err != nil {
		if cursorErr := closeCursor(db); cursorErr != nil {
			oc.logger.Error("failed to close cursor", log.Error(cursorErr))
		}
		return nil, err
	}
	oc.logger.Infof("orders to be collected: %d", totalOrders)
	err = declareCursor(db)
	if err != nil {
		return nil, err
	}
	orderIDs, err := oc.fetchAll(db)
	if cursorErr := closeCursor(db); cursorErr != nil {
		oc.logger.Error("failed to close cursor", log.Error(cursorErr))
	}
	return orderIDs, err
}

func (oc *ordersCollector) fetchAll(db *gorm.DB) ([]string, error) {
	oc.logger.Info("fetching orders from cursor started")
	defer oc.logger.Info("fetching orders from cursor finished")
	orderIDs := make([]string, 0)
	for {
		page, err := fetchPage(oc.config.OrdersFromDatabasePageSize, db)
		if err != nil {
			if cursorErr := closeCursor(db); cursorErr != nil {
				oc.logger.Error("failed to close cursor", log.Error(cursorErr))
			}
			return nil, err
		}
		if len(page) == 0 {
			break
		}
		orderIDs = append(orderIDs, page...)
		oc.logger.Infof("collected %d orders", len(orderIDs))
	}
	return orderIDs, nil
}

func (oc *ordersCollector) sendToLogbroker(ctx context.Context, collectedOrders containers.Set[string]) containers.Set[string] {
	failedToSend := make(containers.Set[string])
	oc.logger.Info("sending orders to logbroker started")
	defer oc.logger.Info("sending orders to logbroker finished")
	sentOrders := 0
	const retriesCount = 1
	for orderID := range collectedOrders {
		if err := oc.unprocessedOrdersService.PutOrderID(ctx, orders.ID(orderID), retriesCount); err != nil {
			oc.logger.Error("failed to send order to logbroker", log.String("orderID", orderID))
			failedToSend.Add(orderID)
			continue
		}
		sentOrders++
		if sentOrders > 0 && sentOrders%oc.config.BatchSize == 0 {
			time.Sleep(oc.config.ThrottleTimeBetweenBatches)
		}
		if sentOrders > 0 && sentOrders%10000 == 0 {
			oc.logger.Infof("%d orders sent to logbroker", sentOrders)
		}
	}
	return failedToSend
}

func extendWithNewCollected(orders []string, previouslyCollected containers.Set[string]) {
	for _, o := range orders {
		previouslyCollected.Add(o)
	}
}

func loadPreviouslyCollectedOrders(file string) (containers.Set[string], error) {
	content, err := ioutil.ReadFile(file)
	if err != nil {
		if os.IsNotExist(err) {
			return containers.SetOf[string](), nil
		}
		return nil, err
	}
	return containers.SetOf(strings.Split(string(content), "\n")...), nil
}

func dumpOrders(file string, orderIDs containers.Set[string]) error {
	list := make([]string, 0, len(orderIDs))
	for o := range orderIDs {
		list = append(list, o)
	}
	return ioutil.WriteFile(file, []byte(strings.Join(list, "\n")), 0666)
}
