package main

import (
	"context"
	stdlog "log"
	"time"

	"github.com/jonboulle/clockwork"
	"golang.yandex/hasql"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/komod/trips/internal/db"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/services/unprocessedorders"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logbroker"
	multilogbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
)

type OrdersDatabaseConfig struct {
	Hosts        []string `config:"ORDERS_DATABASE_HOSTS,required" yaml:"hosts"`
	Port         int      `config:"ORDERS_DATABASE_PORT" yaml:"port"`
	DatabaseName string   `config:"ORDERS_DATABASE_NAME" yaml:"database_name"`
	User         string   `config:"ORDERS_DATABASE_USER" yaml:"user"`
	Password     string   `config:"ORDERS_DATABASE_PASSWORD,required" yaml:"password"`
}

type Config struct {
	OrdersDatabase             OrdersDatabaseConfig     `yaml:"orders_database"`
	InputFile                  string                   `config:"INPUT_FILE" yaml:"input_file"`
	OutputFile                 string                   `config:"OUTPUT_FILE" yaml:"output_file"`
	SendToLogbroker            bool                     `config:"SEND_TO_LOGBROKER" yaml:"send_to_logbroker"`
	UnprocessedOrders          unprocessedorders.Config `yaml:"unprocessed_orders"`
	Debug                      bool                     `yaml:"debug"`
	OrdersFromDatabasePageSize uint                     `yaml:"orders_from_database_page_size"`
	GlobalTimeout              time.Duration            `yaml:"global_timeout"`
	FailedToSendOrdersFile     string                   `yaml:"failed_to_send_orders_file"`
	ThrottleTimeBetweenBatches time.Duration            `yaml:"throttle_time_between_batches"`
	BatchSize                  int                      `yaml:"batch_size"`
}

var defaultConfig = Config{
	OrdersDatabase: OrdersDatabaseConfig{
		Hosts:        []string{},
		Port:         6432,
		DatabaseName: "orders_service",
		User:         "orders_service",
		Password:     "",
	},
	InputFile:                  "orders_to_process.log",
	OutputFile:                 "orders_to_process.log",
	FailedToSendOrdersFile:     "failed_to_send_orders.log",
	SendToLogbroker:            false,
	UnprocessedOrders:          unprocessedorders.DefaultConfig,
	OrdersFromDatabasePageSize: 100000,
	Debug:                      false,
	GlobalTimeout:              12 * time.Hour,
	ThrottleTimeBetweenBatches: 0 * time.Second,
	BatchSize:                  1000,
}

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := defaultConfig
	configLoader := configuration.NewDefaultConfitaLoader()
	err := configLoader.Load(ctx, &config)

	if err != nil {
		stdlog.Fatalf("can not load configuration: %s", err)
	}
	logger, err := logging.NewDeploy(&logging.DefaultConfig)
	if err != nil {
		stdlog.Fatalf("failed to create logger, err: %s", err)
	}
	defer func() {
		err = logger.L.Sync()
		if err != nil {
			stdlog.Println("failed to close logger:", err)
		}
	}()
	pgClient := createPgClient(config.OrdersDatabase, logger)
	var unprocessedOrdersLogbrokerProducer = createUnprocessedOrderLogbrokerProducer(ctx, logger, config.UnprocessedOrders)
	defer unprocessedOrdersLogbrokerProducer.Close()
	unprocessedOrdersStorage := db.NewUnprocessedOrdersStorage(pgClient)
	unprocessedOrdersService := unprocessedorders.NewService(&nop.Logger{}, unprocessedOrdersLogbrokerProducer, clockwork.NewRealClock(), unprocessedOrdersStorage)

	collector := newOrdersCollector(logger, pgClient, unprocessedOrdersService, config)
	collector.collectAndSend(ctx)
}

func createPgClient(config OrdersDatabaseConfig, logger log.Logger) *pgclient.Client {
	pgClient, err := pgclient.NewClientBuilder(
		config.Hosts,
		config.Port,
		config.DatabaseName,
		config.User,
		config.Password,
	).WithClusterOptions(
		hasql.WithUpdateTimeout(1*time.Second),
		hasql.WithUpdateInterval(5*time.Minute),
		hasql.WithTracer(
			hasql.Tracer{
				NodeDead: pgclient.OnDeadNode(logger),
			},
		),
	).Build()
	if err != nil {
		logger.Fatal(err.Error())
	}
	return pgClient
}

func createUnprocessedOrderLogbrokerProducer(ctx context.Context, logger log.Logger, config unprocessedorders.Config) unprocessedorders.LogbrokerProducer {
	if config.Mock {
		return unprocessedorders.NewMockProducer()
	}
	logbrokerProducer, err := logbroker.NewProducer(
		config.Topic,
		config.Endpoint,
		multilogbroker.GetDeploySourceID(config.ProducerID),
		logbroker.NewOAuthCredentialsProvider(config.Token),
		logger,
		logbroker.WithoutSeqNo(),
	)
	if err != nil {
		logger.Fatal("failed to create logbroker producer", log.Error(err))
	}
	err = logbrokerProducer.Run(ctx)
	if err != nil {
		logger.Fatal("failed to run logbroker producer", log.Error(err))
	}
	return logbrokerProducer
}
