package main

import (
	"context"
	"fmt"
	stdlog "log"

	"github.com/jonboulle/clockwork"
	"golang.yandex/hasql"

	"a.yandex-team.ru/library/go/core/log"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/komod/trips/internal/common/dynamicresources"
	"a.yandex-team.ru/travel/komod/trips/internal/common/profiling"
	"a.yandex-team.ru/travel/komod/trips/internal/components/collector"
	"a.yandex-team.ru/travel/komod/trips/internal/db"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/komod/trips/internal/services/processor"
	"a.yandex-team.ru/travel/komod/trips/internal/services/unprocessedorders"
	"a.yandex-team.ru/travel/library/go/configuration"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logbroker"
	multilogbroker "a.yandex-team.ru/travel/library/go/logbroker/multi"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/services/messagecollector"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "travel-trips-collector"

func main() {
	// setting up infrastructure
	maxprocs.AdjustAuto()
	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &collector.Cfg)

	if err != nil {
		stdlog.Fatalf("can not load configuration: %s", err)
	}
	logger, err := logging.NewDeploy(&collector.Cfg.Logging)
	if err != nil {
		stdlog.Fatalf("failed to create logger, err: %s", err)
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			stdlog.Println("failed to close logger:", err)
		}
	}()

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	referencesRegistry, err := references.NewRegistry(collector.Cfg.Dicts)
	if err != nil {
		logger.Fatal("failed to create references registry", log.Error(err))
	}

	runDynamicResourcesService(collector.Cfg, logger, referencesRegistry)

	go func() {
		err = metricsserver.RunMetricsHTTPServer(context.Background(), metricsserver.DefaultMetricsConfig, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	var unprocessedOrdersLogbrokerProducer = createUnprocessedOrderLogbrokerProducer(ctx, logger)
	defer unprocessedOrdersLogbrokerProducer.Close()
	clock := clockwork.NewRealClock()
	pgClient, err := pgclient.NewClientBuilder(
		collector.Cfg.Database.Hosts,
		collector.Cfg.Database.Port,
		collector.Cfg.Database.Name,
		collector.Cfg.Database.User,
		collector.Cfg.Database.Password,
	).WithClusterOptions(
		hasql.WithUpdateTimeout(collector.Cfg.Database.HostsUpdateTimeout),
		hasql.WithTracer(
			hasql.Tracer{
				NodeDead: pgclient.OnDeadNode(logger),
			},
		),
	).Build()
	unprocessedOrdersStorage := db.NewUnprocessedOrdersStorage(pgClient)
	unprocessedOrdersService := unprocessedorders.NewService(
		logger,
		unprocessedOrdersLogbrokerProducer,
		clock,
		unprocessedOrdersStorage,
	)
	go unprocessedOrdersService.MonitorFailedOrdersCount()
	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID:    collector.Cfg.Tvm.SelfAppID,
			WhiteList: collector.Cfg.Tvm.Whitelist,
		},
	)
	connection, err := helpers.CreateConnection(
		collector.Cfg.Processor.YPlannerID,
		collector.Cfg.Processor.TvmID,
		logger,
		&tvmHelper,
	)
	if err != nil {
		logger.Fatal("failed to create connection to processor", log.Error(err))
	}
	defer func() {
		if err := connection.Close(); err != nil {
			logger.Error("failed to close connection to processor", log.Error(err))
		}
	}()

	processorService := processor.NewService(connection, collector.Cfg.Processor)
	collectorService := collector.BuildCollector(logger, collector.Cfg, collector.Cfg.Logbroker, processorService, unprocessedOrdersService)

	runProbesServer(ctx, collectorService, logger, rootRegistry, pgClient)
	runProfilingService()

	collectorService.Run(ctx)
}

func createUnprocessedOrderLogbrokerProducer(ctx context.Context, logger log.Logger) unprocessedorders.LogbrokerProducer {
	if collector.Cfg.UnprocessedOrders.Mock {
		return unprocessedorders.NewMockProducer()
	}
	logbrokerProducer, err := logbroker.NewProducer(
		collector.Cfg.UnprocessedOrders.Topic,
		collector.Cfg.UnprocessedOrders.Endpoint,
		multilogbroker.GetDeploySourceID(collector.Cfg.UnprocessedOrders.ProducerID),
		logbroker.NewOAuthCredentialsProvider(collector.Cfg.UnprocessedOrders.Token),
		logger,
		logbroker.WithoutSeqNo(),
	)
	if err != nil {
		logger.Fatal("failed to create logbroker producer", log.Error(err))
	}
	err = logbrokerProducer.Run(ctx)
	if err != nil {
		logger.Fatal("failed to run logbroker producer", log.Error(err))
	}
	return logbrokerProducer
}

func runProfilingService() {
	profilingService := profiling.NewService(collector.Cfg.Profiling)
	profilingService.BackroundRun()
}

func runDynamicResourcesService(
	cfg collector.Config,
	logger log.Logger,
	dictsRegistry *references.Registry,
) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		cfg.DynamicResources,
		dynamicresources.WithOnUpdateDicts(dictsRegistry.OnUpdate),
	)
	dynamicResourceService.BackgroundRun()
}

func runProbesServer(
	ctx context.Context,
	collectorService *messagecollector.Collector,
	logger log.Logger,
	registry coreMetrics.Registry,
	pgClient *pgclient.Client,
) {
	probesState := probes.NewState(
		logger,
		probes.OnReady(func() error {
			if !collectorService.Ready() {
				return fmt.Errorf("collector not ready")
			}
			return nil
		}),
		probes.OnReady(pgClient.Ping),
	)

	probesServer := httpserver.NewHTTPServerBuilder(
		collector.Cfg.HTTP,
		logger,
		probes.GetChiRouteBuilders(&collector.Cfg.Probes, probesState),
		registry.WithPrefix("probes"),
	).Build()
	go func() {
		_ = probesServer.Run(ctx)
	}()
}
