package main

import (
	"context"
	stdlog "log"

	"golang.yandex/hasql"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/komod/trips/internal/components/processor"
	tripsdb "a.yandex-team.ru/travel/komod/trips/internal/db"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logging"
)

func main() {
	maxprocs.AdjustAuto()
	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &processor.Cfg)

	if err != nil {
		stdlog.Fatalf("can not load configuration: %s", err)
	}
	logger, err := logging.NewDeploy(&processor.Cfg.Logging)
	if err != nil {
		stdlog.Fatalf("failed to create logger, err: %s", err)
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			stdlog.Println("failed to close logger:", err)
		}
	}()

	pgClient, err := pgclient.NewClientBuilder(
		processor.Cfg.Database.Hosts,
		processor.Cfg.Database.Port,
		processor.Cfg.Database.Name,
		processor.Cfg.Database.User,
		processor.Cfg.Database.Password,
	).WithClusterOptions(
		hasql.WithUpdateTimeout(processor.Cfg.Database.HostsUpdateTimeout),
		hasql.WithTracer(
			hasql.Tracer{
				NodeDead: pgclient.OnDeadNode(logger),
			},
		),
	).Build()
	if err != nil {
		logger.Fatal("failed to build pgClient", log.Error(err))
	}

	db, err := pgClient.GetDB(hasql.Primary)
	if err != nil {
		logger.Fatal("failed to take db connection", log.Error(err))
	}
	logger.Info("Start migration")
	if err := db.AutoMigrate(
		tripsdb.AllModels...,
	); err != nil {
		logger.Fatal("failed to migrate trip models", log.Error(err))
	}
	logger.Info("Successful migration")
}
