package main

import (
	"context"
	"fmt"
	stdlog "log"
	"net/http"
	"strconv"

	"golang.yandex/hasql"
	"google.golang.org/grpc"
	ghealth "google.golang.org/grpc/health"
	healthpb "google.golang.org/grpc/health/grpc_health_v1"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/collect/policy/inflight"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/library/go/yandex/tvm/cachedtvm"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
	"a.yandex-team.ru/travel/avia/library/go/probes"
	"a.yandex-team.ru/travel/komod/trips/internal/common/dynamicresources"
	"a.yandex-team.ru/travel/komod/trips/internal/common/profiling"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api"
	"a.yandex-team.ru/travel/komod/trips/internal/components/processor"
	eventlogger "a.yandex-team.ru/travel/komod/trips/internal/components/processor/eventslogger"
	processorgrpc "a.yandex-team.ru/travel/komod/trips/internal/components/processor/grpc"
	"a.yandex-team.ru/travel/komod/trips/internal/db"
	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/clients"
	"a.yandex-team.ru/travel/komod/trips/internal/orders/mappers"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	"a.yandex-team.ru/travel/komod/trips/internal/trips/matcher"
	"a.yandex-team.ru/travel/komod/trips/internal/unifiedagent"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/geobase"
	grpcserver "a.yandex-team.ru/travel/library/go/grpcutil/server"
	httpserver "a.yandex-team.ru/travel/library/go/httputil/server"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"a.yandex-team.ru/travel/library/go/syncutil"
	"a.yandex-team.ru/travel/library/go/tracing"
	tvmutil "a.yandex-team.ru/travel/library/go/tvm"
)

const serviceName = "travel-trips-processor"

func main() {
	maxprocs.AdjustAuto()
	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &processor.Cfg)

	if err != nil {
		stdlog.Fatalf("can not load configuration: %s", err)
	}
	logger, err := logging.NewDeploy(&processor.Cfg.Logging)
	if err != nil {
		stdlog.Fatalf("failed to create logger, err: %s", err)
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			stdlog.Println("failed to close logger:", err)
		}
	}()

	tracerCloser := tracing.InitializeDefaultTracer(serviceName)
	defer func() {
		err = tracerCloser.Close()
		if err != nil {
			logger.Fatal("tracer close error:", log.Error(err))
		}
	}()

	rootRegistry := metrics.NewRegistryWithDeployTagsAndExplicitHost()
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	metrics.RunPerfMetricsUpdater(rootRegistry, metricsserver.DefaultMetricsConfig.PerfMetricsRefreshInterval)

	referencesRegistry, err := references.NewRegistry(processor.Cfg.Dicts)
	if err != nil {
		logger.Fatal("failed to create references registry", log.Error(err))
	}
	settlementByStationExtractor := extractors.NewSettlementByStationExtractor(referencesRegistry)

	geoBase, err := geobase.NewGeobase(processor.Cfg.Geobase, logger)
	if err != nil {
		logger.Fatal("failed to create geobase", log.Error(err))
	}
	defer geoBase.Destroy()

	runDynamicResourcesService(logger, referencesRegistry)

	go func() {
		err = metricsserver.RunMetricsHTTPServer(context.Background(), metricsserver.DefaultMetricsConfig, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	pgClient, err := pgclient.NewClientBuilder(
		processor.Cfg.Database.Hosts,
		processor.Cfg.Database.Port,
		processor.Cfg.Database.Name,
		processor.Cfg.Database.User,
		processor.Cfg.Database.Password,
	).WithPoolOptions(
		pgclient.MaxOpenPoolConns(200),
	).WithClusterOptions(
		hasql.WithUpdateTimeout(processor.Cfg.Database.HostsUpdateTimeout),
		hasql.WithTracer(
			hasql.Tracer{
				UpdatedNodes: pgclient.OnUpdatedNodes,
				NodeDead:     pgclient.OnDeadNode(logger),
			},
		),
	).Build()
	if err != nil {
		logger.Fatal("failed to build pgClient", log.Error(err))
	}

	var tvmClient tvm.Client
	tvmAllowedIds := tvmutil.TvmClientIDFromInt(processor.Cfg.Tvm.Whitelist)
	if processor.Cfg.Tvm.Enabled {
		tvmClient, err = tvmtool.NewDeployClient(tvmtool.WithSrc(strconv.Itoa(int(processor.Cfg.Tvm.SelfAppID))))
		if err != nil {
			logger.Fatal("failed to create tvm client", log.Error(err))
		}
		tvmClient, err = cachedtvm.NewClient(
			tvmClient,
			cachedtvm.WithCheckServiceTicket(
				processor.Cfg.Tvm.CacheTTL,
				processor.Cfg.Tvm.CacheMaxItems,
			),
		)
		if err != nil {
			logger.Fatal("failed to create cached tvm client", log.Error(err))
		}
	}

	cachedLocationRepository := helpers.NewCachedLocationRepository()
	pointFactory := point.NewFactory(
		extractors.NewStationIDToSettlementIDMapper(referencesRegistry),
		cachedLocationRepository,
		geoBase,
		referencesRegistry,
	)
	pointResolver := point.NewResolver(geoBase, referencesRegistry, pointFactory)
	pointComparator := point.NewComparator(geoBase, referencesRegistry, pointResolver)
	spanComparator := span.NewSpanComparator(pointComparator)
	spansHelper := span.NewHelper(pointComparator, spanComparator)
	matcherRuleFactory := matcher.NewRuleFactory(pointComparator, spansHelper)

	ordersHTTPClient := clients.NewHTTPClient(
		processor.Cfg.OrdersClient,
		&http.Client{Timeout: processor.Cfg.OrdersClient.RequestTimeout},
		tvmClient,
	)
	ordersClient := clients.NewMappingClient(
		logger,
		ordersHTTPClient,
		mappers.NewAviaOrderMapper(referencesRegistry, cachedLocationRepository, settlementByStationExtractor),
		mappers.NewHotelOrderMapper(geoBase, referencesRegistry, pointFactory),
		mappers.NewTrainOrderMapper(referencesRegistry, cachedLocationRepository, settlementByStationExtractor),
		mappers.NewBusOrderMapper(referencesRegistry, cachedLocationRepository, settlementByStationExtractor),
	)

	unifiedAgentClient, err := unifiedagent.NewClient(processor.Cfg.UnifiedAgent, ctx, logger)
	if err != nil {
		logger.Fatal("failed to create unified agent client", log.Error(err))
	}
	unifiedAgentClient.CollectMetrics(rootRegistry.WithPrefix("unifiedagent"), inflight.NewCollectorPolicy())
	defer unifiedAgentClient.Close(ctx)
	eventsLogger := eventlogger.NewEventsLogger(logger, unifiedAgentClient)
	processorComponent := processor.NewProcessor(
		logger,
		ordersClient,
		extractors.NewOrderInfoExtractor(pointFactory),
		matcher.NewMatcher(matcherRuleFactory.Make(), processor.Cfg.Matcher),
		db.NewTripsStorage(processor.Cfg.TransactionOptions, pgClient, pointFactory),
		eventsLogger,
	)
	processorService := processorgrpc.NewService(processorComponent, processor.Cfg.ProcessingTimeout)
	healthServer := ghealth.NewServer()
	healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_SERVING)
	runProbesServer(ctx, logger, healthServer, pgClient, rootRegistry)
	runProfilingService()

	grpcServiceRegisterers := []grpcserver.ServiceRegisterer{
		processorService.GetServiceRegisterer(),
	}

	wg := syncutil.WaitGroup{}
	wg.Go(
		func() {
			server := grpcserver.NewGrpcServerBuilder(
				processor.Cfg.Grpc,
				logger.WithName("servers.grpc").(*zap.Logger),
			).
				WithRegisterers(grpcServiceRegisterers...).
				WithOptionalTVM(tvmClient, tvmAllowedIds).
				WithHealthServer(healthServer).
				WithInterceptors(helpers.MakeMetricsInterceptor(rootRegistry.WithPrefix("grpc"))).
				Build()
			err = server.Run(context.Background())
			if err != nil {
				logger.Fatal("Error while starting Grpc server", log.Error(err))
			}
		},
	)

	logger.Info("Started")
	wg.Wait()
}

func runProfilingService() {
	profilingService := profiling.NewService(processor.Cfg.Profiling)
	profilingService.BackroundRun()
}

func runDynamicResourcesService(logger log.Logger, dictsRegistry *references.Registry) {
	dynamicResourceService := dynamicresources.NewService(
		logger,
		processor.Cfg.DynamicResources,
		dynamicresources.WithOnUpdateDicts(dictsRegistry.OnUpdate),
	)
	dynamicResourceService.BackgroundRun()
}

func runProbesServer(ctx context.Context, logger log.Logger, healthServer *ghealth.Server, pgClient *pgclient.Client, registry coreMetrics.Registry) {
	tvmHelper := tvmutil.NewDeployTvmHelper(
		logger,
		&tvmutil.TvmHelperConfig{
			SelfID:    processor.Cfg.Tvm.SelfAppID,
			WhiteList: processor.Cfg.Tvm.Whitelist,
		},
	)
	tvmInterceptor := tvmHelper.GRPCClientInterceptor(processor.Cfg.Tvm.SelfAppID)
	if tvmInterceptor == nil {
		logger.Error("failed to create tvm interceptor for readiness server")
		return
	}
	onReady := func() error {
		dialOptions := []grpc.DialOption{
			grpc.WithInsecure(),
			grpc.WithBlock(),
		}
		if !api.Cfg.IsDevelopment() {
			logger.Info("add tvm interceptor to readiness server")
			dialOptions = append(dialOptions, grpc.WithUnaryInterceptor(tvmInterceptor))
		}

		clientConn, err := grpc.DialContext(
			ctx,
			api.Cfg.Grpc.Addr,
			dialOptions...,
		)
		if err != nil {
			return fmt.Errorf("unable to dial connection: %w", err)
		}
		defer clientConn.Close()
		response, err := healthpb.NewHealthClient(clientConn).Check(
			ctx,
			&healthpb.HealthCheckRequest{Service: serviceName},
		)
		if err != nil {
			return fmt.Errorf("failed to check health service: %w", err)
		}
		if response.GetStatus() != healthpb.HealthCheckResponse_SERVING {
			return fmt.Errorf("failed to check health service")
		}
		return nil
	}
	onStop := func() {
		healthServer.SetServingStatus(serviceName, healthpb.HealthCheckResponse_NOT_SERVING)
	}

	probesState := probes.NewState(
		logger,
		probes.OnReady(onReady),
		probes.OnReady(pgClient.Ping),
		probes.OnStopBefore(onStop),
	)
	probesServer := httpserver.NewHTTPServerBuilder(
		processor.Cfg.HTTP,
		logger,
		probes.GetChiRouteBuilders(&processor.Cfg.Probes, probesState),
		registry.WithPrefix("probes"),
	).Build()
	go func() {
		_ = probesServer.Run(ctx)
	}()
}
