package dynamicresources

import (
	"encoding/json"
	"net/http"
	"sync"

	"github.com/go-chi/chi/v5"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/httputil/headers"
)

type Config struct {
	Addr             string
	UpdateDictsRoute string
}

var DefaultConfig = Config{
	Addr:             "[::]:9006",
	UpdateDictsRoute: "/update_dicts",
}

type Service struct {
	mutex          sync.Mutex
	logger         log.Logger
	onUpdateDicts  func() error
	onUpdateHotels func() error
	config         Config
}

type dynamicResourceOption func(service *Service)

func WithOnUpdateDicts(onUpdateDicts func() error) dynamicResourceOption {
	return func(service *Service) {
		service.onUpdateDicts = onUpdateDicts
	}
}

func NewService(logger log.Logger, config Config, options ...dynamicResourceOption) *Service {
	service := &Service{
		mutex:          sync.Mutex{},
		logger:         logger.WithName("DynamicResourcesService"),
		config:         config,
		onUpdateDicts:  func() error { return nil },
		onUpdateHotels: func() error { return nil },
	}
	for _, opt := range options {
		opt(service)
	}
	return service
}

func (s *Service) BackgroundRun() {
	router := chi.NewRouter()
	router.Get(s.config.UpdateDictsRoute, s.updateDictsHandler)
	go func() {
		_ = http.ListenAndServe(s.config.Addr, router)
	}()
}

type updateResourcesResponse struct {
	Status string `json:"status"`
	Error  string `json:"error,omitempty"`
}

var successResponse, _ = json.Marshal(updateResourcesResponse{Status: "success"})

func (s *Service) updateDictsHandler(w http.ResponseWriter, r *http.Request) {
	s.logger.Info("update shared dicts dynamic resource request")
	s.mutex.Lock()
	defer s.mutex.Unlock()
	w.WriteHeader(http.StatusOK)
	w.Header().Set(headers.ContentTypeKey, headers.TypeApplicationJSON.String())
	if err := s.onUpdateDicts(); err == nil {
		_, _ = w.Write(successResponse)
		s.logger.Info("shared dicts dynamic resource updated successfully")
	} else {
		s.logger.Error("failed to update shared dicts", log.Error(err))
		response, _ := json.Marshal(updateResourcesResponse{Status: "failed", Error: err.Error()})
		_, _ = w.Write(response)
	}
}
