package async

import (
	"context"
	"fmt"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/x/math"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/helpers/http"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/services/activities"
	activitiesmodels "a.yandex-team.ru/travel/komod/trips/internal/services/activities/models"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	tripmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/library/go/syncutil"
)

const (
	avtivityProviderName = "api.trips.async.BlockProvider"

	IziTravelResultsLimit = 10
	AfishaResultsLimit    = 10
	AfishaRawResultsLimit = 20
	AfishaPeriod          = 5
)

type ActivitiesProvider struct {
	logger          log.Logger
	spanHelper      *span.Helper
	afishaClient    activities.AfishaClient
	iziTravelClient activities.IziTravelClient
	pointResolver   *point.Resolver
}

func NewActivitiesProvider(
	logger log.Logger,
	spanHelper *span.Helper,
	afishaClient activities.AfishaClient,
	iziTravelClient activities.IziTravelClient,
	pointResolver *point.Resolver,
) *ActivitiesProvider {
	return &ActivitiesProvider{
		logger:          logger,
		spanHelper:      spanHelper,
		afishaClient:    afishaClient,
		iziTravelClient: iziTravelClient,
		pointResolver:   pointResolver,
	}
}

func (p *ActivitiesProvider) GetEmptyBlock() apimodels.AsyncBlockRsp {
	return apimodels.ActivitiesBlock{}
}

func (p *ActivitiesProvider) Available(trip *tripmodels.Trip) bool {
	return p.getFirstVisitWithGeoID(trip) != nil
}

func (p *ActivitiesProvider) GetBlock(
	ctx context.Context,
	trip *tripmodels.Trip,
) (_ apimodels.AsyncBlockRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetBlock", avtivityProviderName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	visitWithActivities := p.getFirstVisitWithGeoID(trip)
	if visitWithActivities == nil {
		return nil, fmt.Errorf("activities not available")
	}

	targetPoint := visitWithActivities.Point()
	geoID := targetPoint.GetGeoID()

	var afishaEvents []activitiesmodels.AfishaEventInfo
	var iziTravelTours []activitiesmodels.IziTravelTourInfo

	var wg syncutil.WaitGroup
	if p.isAfishaEnabledForPoint(targetPoint) {
		wg.Go(func() {
			afishaEvents = p.fetchAfishaEvents(ctx, visitWithActivities.When(), geoID)
		})
	}
	wg.Go(func() {
		iziTravelTours = p.fetchIziTravelTours(ctx, geoID)
	})
	wg.Wait()

	if len(afishaEvents) == 0 && len(iziTravelTours) == 0 {
		return apimodels.ActivitiesBlock{Blocks: []apimodels.Activity{}}, nil
	}
	zippedActivities := zipActivities(
		mapAfishaActivities(afishaEvents),
		mapIziTravelActivities(iziTravelTours),
	)
	return apimodels.ActivitiesBlock{
		Blocks: zippedActivities,
	}, nil
}

func (p *ActivitiesProvider) fetchIziTravelTours(ctx context.Context, geoID int) []activitiesmodels.IziTravelTourInfo {
	iziTravelTours, err := p.iziTravelClient.GetActivities(
		ctx,
		IziTravelResultsLimit,
		geoID,
	)
	if err != nil {
		if _, isNotFound := err.(*http.NotFoundErr); isNotFound {
			return []activitiesmodels.IziTravelTourInfo{}
		}
		p.logger.Error(
			"unable to get izi-travel activities",
			log.Error(err),
			log.Int("geo_id", geoID),
		)
	}
	return iziTravelTours
}

func (p *ActivitiesProvider) fetchAfishaEvents(ctx context.Context, when time.Time, geoID int) []activitiesmodels.AfishaEventInfo {
	afishaEvents, err := p.afishaClient.GetActivities(
		ctx,
		when,
		AfishaResultsLimit,
		AfishaRawResultsLimit,
		AfishaPeriod,
		geoID,
	)
	if err != nil {
		if _, isNotFound := err.(*http.NotFoundErr); isNotFound {
			return []activitiesmodels.AfishaEventInfo{}
		}
		p.logger.Error(
			"unable to get afisha events",
			log.Error(err),
			log.String("start_date", when.String()),
			log.Int("geo_id", geoID),
		)
	}
	return afishaEvents
}

func (p *ActivitiesProvider) getFirstVisitWithGeoID(trip *tripmodels.Trip) *models.Visit {
	if trip == nil {
		return nil
	}

	spans := p.spanHelper.ReduceTransfers(trip.GetActiveSpans())
	visits := p.spanHelper.ExtractVisitsRemovingExtremes(spans)

	now := time.Now()
	for _, visit := range visits {
		geoID := visit.Point().GetGeoID()
		if visit.When().After(now) && geoID != 0 {
			return &visit
		}
	}
	return nil
}

func (p *ActivitiesProvider) isAfishaEnabledForPoint(point models.Point) bool {
	countryPoint, err := p.pointResolver.GetCountry(point)
	if err != nil {
		return false
	}
	return countryPoint.GetGeoID() == consts.RussiaGeoID
}

func mapAfishaActivities(events []activitiesmodels.AfishaEventInfo) []apimodels.Activity {
	result := make([]apimodels.Activity, 0, len(events))
	for _, event := range events {
		result = append(
			result,
			&apimodels.AfishaEventInfo{
				Name:     event.Name,
				MinPrice: mapPrice(event.MinPrice),
				Type:     event.Type,
				DateTime: event.DateTime,
				DateText: event.DateText,
				ImageURL: event.ImageURL,
				EventURL: event.EventURL,
				Tags:     event.Tags,
			},
		)
	}
	return result
}

func mapPrice(price *activitiesmodels.Price) *apimodels.Price {
	if price == nil {
		return nil
	}
	return &apimodels.Price{
		Value:    price.Value,
		Currency: price.Currency,
	}
}

func mapIziTravelActivities(tours []activitiesmodels.IziTravelTourInfo) []apimodels.Activity {
	result := make([]apimodels.Activity, 0, len(tours))
	for _, tour := range tours {
		result = append(
			result,
			&apimodels.IziTravelTourInfo{
				Name:     tour.Name,
				ImageURL: tour.ImageURL,
				TourURL:  tour.TourURL,
				Type:     tour.Type,
				Category: tour.Category,
				Duration: tour.Duration,
			},
		)
	}
	return result
}

func zipActivities(activityChunks ...[]apimodels.Activity) []apimodels.Activity {
	var allActivityCount int
	var chunkMaxCount int
	for _, chunk := range activityChunks {
		allActivityCount += len(chunk)
		chunkMaxCount = math.MaxInt(chunkMaxCount, len(chunk))
	}
	result := make([]apimodels.Activity, 0, allActivityCount)
	for i := 0; i < chunkMaxCount; i++ {
		for _, chunk := range activityChunks {
			if i >= len(chunk) {
				continue
			}
			result = append(result, chunk[i])
		}
	}
	return result
}
