package async

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/errutil"
)

const (
	blockProviderName = "api.trips.async.BlockProvider"
)

type subProvider interface {
	Available(*models.Trip) bool
	GetEmptyBlock() apimodels.AsyncBlockRsp
	GetBlock(context.Context, *models.Trip) (apimodels.AsyncBlockRsp, error)
}

type BlockProvider struct {
	logger             log.Logger
	activitiesProvider subProvider
}

func NewBlockProvider(
	logger log.Logger,
	activitiesProvider *ActivitiesProvider,
) *BlockProvider {
	return &BlockProvider{
		logger:             logger,
		activitiesProvider: activitiesProvider,
	}
}

func (p BlockProvider) BlockIsAvailable(
	blockType BlockType,
	trip *models.Trip,
) bool {
	provider, err := p.getSubProviderByBlockType(blockType)
	if err != nil {
		panic(err)
	}
	return provider.Available(trip)
}

func (p BlockProvider) GetEmptyBlockForType(blockType BlockType) apimodels.AsyncBlockRsp {
	provider, err := p.getSubProviderByBlockType(blockType)
	if err != nil {
		panic(err)
	}
	return provider.GetEmptyBlock()
}

func (p BlockProvider) GetBlocks(
	ctx context.Context,
	blockTypes []BlockType,
	trip *models.Trip,
) (_ []apimodels.AsyncBlockRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetBlocks", blockProviderName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	var result []apimodels.AsyncBlockRsp
	for _, blockType := range blockTypes {
		provider, err := p.getSubProviderByBlockType(blockType)
		if err != nil {
			return nil, err
		}
		block, err := provider.GetBlock(ctx, trip)
		if err != nil {
			return nil, err
		}
		result = append(result, block)
	}
	return result, nil
}

func (p BlockProvider) getSubProviderByBlockType(blockType BlockType) (subProvider, error) {
	switch blockType {
	case ActivitiesBlock:
		return p.activitiesProvider, nil
	default:
		return nil, fmt.Errorf("unexpected async block type %s", blockType)
	}

}
