package trips

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	tripmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
)

const (
	totalHotelLimit        = 10
	allowedCrossSaleNights = 30
)

type CrossSaleProvider struct {
	logger     log.Logger
	spanHelper *span.Helper
}

func NewCrossSaleProvider(logger log.Logger, spanHelper *span.Helper) *CrossSaleProvider {
	return &CrossSaleProvider{logger: logger, spanHelper: spanHelper}
}

func (p *CrossSaleProvider) MakeBlock(trip *tripmodels.Trip) *apimodels.HotelsCrossSale {
	if trip == nil {
		return nil
	}

	spans := trip.GetActiveSpans()
	if !p.blockIsAvailable(spans) {
		return nil
	}

	spans = p.spanHelper.ReduceTransfers(spans)
	data := p.prepareCrossSaleDataForGaps(spans)
	if data == nil && !p.spanHelper.IsRoundTrip(spans...) {
		data = p.prepareCrossSaleDataForTripWithoutGaps(spans)
	}
	return p.makeBlockFromData(data)
}

type preparedCrossSaleData struct {
	targetVisit models.Visit
	days        time.Duration
}

func (p *CrossSaleProvider) makeBlockFromData(data *preparedCrossSaleData) *apimodels.HotelsCrossSale {
	if data == nil {
		return nil
	}

	startCrossSaleDate := data.targetVisit.When()
	return &apimodels.HotelsCrossSale{
		Title:              p.getTitle(data.targetVisit.Point()),
		CheckInDate:        startCrossSaleDate,
		CheckOutDate:       startCrossSaleDate.Add(data.days * consts.Day),
		Adults:             1,
		SettlementPointKey: data.targetVisit.Point().GetPointKey(),
		TotalHotelLimit:    totalHotelLimit,
	}
}

func (p *CrossSaleProvider) getTitle(point models.Point) string {
	pointLinguistics := point.GetLinguistics()
	preposition := pointLinguistics[models.Preposition]
	if preposition == "" {
		preposition = "в"
	}
	title := pointLinguistics[models.PrepositionalCase]
	if title == "" {
		p.logger.Error(
			"empty point title",
			log.String("form", models.PrepositionalCase.String()),
			log.String("point_key", point.GetPointKey()),
		)
		return "Выберите подходящий отель"
	}

	return fmt.Sprintf(
		"Выберите подходящий отель %s %s",
		preposition,
		title,
	)
}

func (p *CrossSaleProvider) prepareCrossSaleDataForTripWithoutGaps(spans []models.Span) *preparedCrossSaleData {
	if len(spans) == 0 {
		return nil
	}
	right := p.spanHelper.GetRightMostSpan(spans...)
	if right.End().When().Before(time.Now()) {
		return nil
	}

	return &preparedCrossSaleData{
		targetVisit: right.End(),
		days:        1,
	}
}

// prepareCrossSaleDataForTripWithoutGaps для каждой дыры патаемся подготовить данные для кроссейла
func (p *CrossSaleProvider) prepareCrossSaleDataForGaps(spans []models.Span) *preparedCrossSaleData {
	if len(spans) == 0 {
		return nil
	}

	gaps := p.spanHelper.MakeGaps(spans)
	gaps = excludePastSpans(gaps)
	if len(gaps) == 0 {
		return nil
	}

	p.spanHelper.SortByStart(gaps)
	for _, gap := range gaps {
		data := p.prepareCrossSaleDataFromGap(gap)
		if data != nil {
			return data
		}
	}
	return nil
}

// prepareCrossSaleDataFromGap если дыра начинается в прошлом то предлагаем отель с текущей даты
// ничего не предлагаем если дыра меньше дня
func (p *CrossSaleProvider) prepareCrossSaleDataFromGap(gap models.Span) *preparedCrossSaleData {
	targetVisit := gap.Start()
	now := time.Now()
	if targetVisit.When().Before(now) {
		targetVisit = models.NewVisit(targetVisit.Point(), now)
	}

	days := gap.End().When().Sub(targetVisit.When()) / consts.Day
	if days < 1 {
		return nil
	}
	if days > allowedCrossSaleNights {
		days = allowedCrossSaleNights
	}
	return &preparedCrossSaleData{
		targetVisit: targetVisit,
		days:        days,
	}
}

func (p *CrossSaleProvider) blockIsAvailable(spans []models.Span) bool {
	if hasHotelSpans(spans) {
		return false
	}
	if len(spans) == 0 {
		return false
	}

	rightMostVisit := p.spanHelper.GetRightMostSpan(spans...).End()
	return !rightMostVisit.When().Before(time.Now())
}

func excludePastSpans(spans []models.Span) []models.Span {
	result := make([]models.Span, 0)
	now := time.Now()
	for _, s := range spans {
		if s.End().When().After(now) {
			result = append(result, s)
		}
	}
	return result
}

func hasHotelSpans(spans []models.Span) bool {
	for _, s := range spans {
		if !s.IsTransport() {
			return true
		}
	}
	return false
}
