package displaytime

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/units"
)

const useLongMonthName = true

var monthShortNames = []string{
	"янв",
	"фев",
	"мар",
	"апр",
	"мая",
	"июн",
	"июл",
	"авг",
	"сен",
	"окт",
	"ноя",
	"дек",
}

var monthLongNames = []string{
	"января",
	"февраля",
	"марта",
	"апреля",
	"мая",
	"июня",
	"июля",
	"августа",
	"сентября",
	"октября",
	"ноября",
	"декабря",
}

type Builder struct{}

func NewBuilder() *Builder {
	return &Builder{}
}

func (b Builder) BuildDatesPair(args *DatesPairArgs) string {
	if args.useRelative {
		if displayDate, ok := b.buildRelativeDate(args.userTime, args.start); ok {
			return displayDate
		}
	}

	end := datePart{
		needYear:  args.useYear,
		needMonth: true,
		needDay:   true,
		date:      args.end,
	}

	start := datePart{
		needYear:  args.start.Year() != args.end.Year() && args.useYear,
		needMonth: args.start.Month() != args.end.Month(),
		needDay:   args.start.Day() != args.end.Day(),
		date:      args.start,
	}

	isLongMonthName := !start.needMonth && !args.shortMonth

	if !start.needDay && !start.needMonth && !start.needYear {
		return b.formatDatePart(end, isLongMonthName)
	}

	parts := [2]string{b.formatDatePart(start, isLongMonthName), b.formatDatePart(end, isLongMonthName)}
	if !start.needMonth && !start.needYear {
		return fmt.Sprintf("%s — %s", parts[0], parts[1])
	} else {
		return fmt.Sprintf("%s — %s", parts[0], parts[1])
	}
}

func (b Builder) buildRelativeDate(userTime, t time.Time) (string, bool) {
	userNow := userTime.Truncate(units.Day)
	start := t.In(userTime.Location()).Truncate(units.Day)

	switch start.Sub(userNow) {
	case 0:
		return "Сегодня", true
	case units.Day:
		return "Завтра", true
	case 2 * units.Day:
		return "Послезавтра", true
	default:
		return "", false
	}
}

type datePart struct {
	needTime  bool
	needYear  bool
	needMonth bool
	needDay   bool
	date      time.Time
}

func (b Builder) formatDatePart(part datePart, isLongMonthName bool) string {
	monthMap := monthShortNames
	if isLongMonthName {
		monthMap = monthLongNames
	}

	monthName := monthMap[int(part.date.Month())-1]
	if part.needYear {
		return fmt.Sprintf("%d %s, %d",
			part.date.Day(),
			monthName,
			part.date.Year(),
		)
	}
	if part.needTime {
		return fmt.Sprintf("%d %s, %s",
			part.date.Day(),
			monthName,
			part.date.Format("15:04"),
		)
	}
	if part.needMonth {
		return fmt.Sprintf("%d %s",
			part.date.Day(),
			monthName,
		)
	}
	return fmt.Sprintf("%d", part.date.Day())
}

func (b Builder) BuildDateTime(args *DateTimeArgs) string {
	if args.time == nil {
		return ""
	}

	if args.useRelative {
		if displayDate, ok := b.buildRelativeDate(args.userTime, *args.time); ok {
			return displayDate
		}
	}

	part := datePart{
		needTime:  true,
		needMonth: true,
		needDay:   true,
		date:      *args.time,
	}
	return b.formatDatePart(part, useLongMonthName)
}
