package grpc

import (
	"context"
	"errors"
	"fmt"
	"time"

	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/grpc/status"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	commons "a.yandex-team.ru/travel/komod/trips/api"
	"a.yandex-team.ru/travel/komod/trips/api/processor/v1"
	"a.yandex-team.ru/travel/komod/trips/api/trips/v1"
	api "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/async"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/usercredentials"
	travelcommonsproto "a.yandex-team.ru/travel/proto"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type UnprocessedOrderHandler interface {
	Put(context.Context, *processor.ProcessTripEntityReq) error
}

type Service struct {
	logger                  log.Logger
	provider                *api.Provider
	unprocessedOrderHandler UnprocessedOrderHandler
}

func NewService(logger log.Logger, provider *api.Provider, unprocessedOrderHandler UnprocessedOrderHandler) *Service {
	return &Service{logger: logger, provider: provider, unprocessedOrderHandler: unprocessedOrderHandler}
}

func (s *Service) GetActiveTrips(ctx context.Context, req *trips.GetActiveTripsReq) (*trips.GetActiveTripsRsp, error) {
	activeTrips, err := s.provider.GetActiveTrips(ctx, req.GetPassportId(), uint(req.GetLimit()), int(req.GetGeoId()))
	if err != nil {
		return nil, fmt.Errorf("api.grpc.Service: %w", err)
	}

	return &trips.GetActiveTripsRsp{
		Trips: s.mapTripsList(activeTrips),
	}, nil
}

func (s *Service) GetTrips(ctx context.Context, req *trips.GetTripsReq) (*trips.GetTripsRsp, error) {
	response, err := s.provider.GetTrips(ctx, req.PassportId, uint(req.GetPastLimit()), int(req.GetGeoId()))
	if err != nil {
		return nil, fmt.Errorf("api.grpc.Service.GetTrips: %w", err)
	}
	return &trips.GetTripsRsp{
		Active: s.mapPaginatedTripsList(response.Active),
		Past:   s.mapPaginatedTripsList(response.Past),
	}, nil
}

func (s *Service) GetMoreTrips(ctx context.Context, req *trips.GetMoreTripsReq) (*trips.GetMoreTripsRsp, error) {
	page, err := s.provider.GetMoreTrips(
		ctx,
		req.PassportId,
		req.ContinuationToken,
		uint(req.Limit),
		int(req.GetGeoId()),
	)
	if err != nil {
		return nil, fmt.Errorf("api.grpc.Service.GetMoreTrips: %w", err)
	}

	return &trips.GetMoreTripsRsp{
		Trips: s.mapPaginatedTripsList(page),
	}, nil
}

func (s *Service) GetTrip(ctx context.Context, req *trips.GetTripReq) (*trips.GetTripRsp, error) {
	var trip *models.TripRsp
	var err error

	trip, err = s.provider.GetTrip(ctx, req.PassportId, req.TripId, int(req.GetGeoId()))
	if err != nil {
		if errors.Is(err, api.ErrTripNotFound) {
			return nil, status.Error(codes.NotFound, fmt.Sprintf("no trip for id %s", req.TripId))
		}
		if errors.Is(err, api.ErrTripIsForbidden) {
			return nil, status.Error(
				codes.PermissionDenied,
				fmt.Sprintf(
					"forbidden trip for id %s and passportID %s",
					req.TripId,
					req.PassportId,
				),
			)
		}
		return nil, fmt.Errorf("api.grpc.Service.GetTrip: %w", err)
	}

	return &trips.GetTripRsp{
		Trip: &trips.Trip{
			Id:          trip.ID,
			Image:       &commons.Image{Url: trip.Image},
			BeginDate:   mapProtoDate(trip.BeginDate),
			EndDate:     mapProtoDate(trip.EndDate),
			DisplayDate: trip.DisplayDate,
			Title:       trip.Title,
			Blocks:      s.mapBlocks(ctx, trip.Blocks),
			State:       trip.State,
		},
	}, nil
}

//nolint:ST1003
func (s *Service) GetTripIdByOrderId(
	ctx context.Context,
	req *trips.GetTripIdByOrderIdReq,
) (*trips.GetTripIdByOrderIdRsp, error) {
	passportID, err := s.getPassportID(ctx)
	if err != nil {
		return nil, err
	}

	tripID, err := s.provider.GetTripIDByOrderID(ctx, passportID, req.GetOrderId())
	if err != nil {
		if errors.Is(err, api.ErrTripNotFound) {
			return nil, status.Error(codes.NotFound, fmt.Sprintf("no trip for orderId %s", req.GetOrderId()))
		}
		return nil, fmt.Errorf("api.grpc.Service: %w", err)
	}

	return &trips.GetTripIdByOrderIdRsp{
		TripId: tripID,
	}, nil
}

func (s *Service) GetTripAsyncBlocks(
	ctx context.Context,
	req *trips.GetTripAsyncBlocksReq,
) (*trips.GetTripAsyncBlocksRsp, error) {
	passportID, err := s.getPassportID(ctx)
	if err != nil {
		return nil, err
	}

	asyncBlockTypes, err := mapAsyncBlockTypes(req.GetBlockTypes())
	if err != nil {
		return nil, status.Error(codes.InvalidArgument, err.Error())
	}

	asyncBlocks, err := s.provider.GetTripAsyncBlocks(ctx, passportID, req.GetTripId(), asyncBlockTypes)
	if err != nil {
		if errors.Is(err, api.ErrTripNotFound) {
			return nil, status.Error(codes.NotFound, fmt.Sprintf("trip not found by id %s", req.GetTripId()))
		}
		if errors.Is(err, api.ErrTripIsForbidden) {
			return nil, status.Error(
				codes.PermissionDenied,
				fmt.Sprintf(
					"forbidden trip for id %s and passportID %s",
					req.TripId,
					passportID,
				),
			)
		}
		return nil, fmt.Errorf("api.grpc.Service: %w", err)
	}

	return &trips.GetTripAsyncBlocksRsp{
		Blocks: s.mapAsyncBlocks(ctx, asyncBlocks),
	}, nil
}

func mapAsyncBlockTypes(protoBlockTypes []trips.BlockType) ([]async.BlockType, error) {
	if len(protoBlockTypes) == 0 {
		return nil, fmt.Errorf("expected non zero number of block")
	}

	var result []async.BlockType
	for _, protoBlockType := range protoBlockTypes {
		var asyncBlockType async.BlockType

		switch protoBlockType {
		case trips.BlockType_ACTIVITIES_BLOCK_TYPE:
			asyncBlockType = async.ActivitiesBlock
		default:
			return nil, fmt.Errorf("unexpected async block type %s", protoBlockType.String())
		}
		result = append(result, asyncBlockType)
	}
	return result, nil
}

func (s *Service) getPassportID(ctx context.Context) (string, error) {
	userCredentials := usercredentials.FromContext(ctx)
	passportID, ok := userCredentials.PassportID()
	if !ok {
		return "", status.Error(
			codes.Unauthenticated,
			fmt.Sprintf(
				"%s not found in metadata",
				usercredentials.UserCredentialPassportID,
			),
		)
	}
	return passportID, nil
}

func (s *Service) mapPaginatedTripsList(response *models.PaginatedTripsListRsp) *trips.PaginatedTripsList {
	return &trips.PaginatedTripsList{
		Trips:             s.mapTripsList(response.Trips),
		ContinuationToken: response.ContinuationToken,
	}
}

func (s *Service) mapTripsList(items []models.TripItemRsp) (result []*trips.TripsListItem) {
	for _, item := range items {
		result = append(result, s.mapTripsListItem(item))
	}
	return result
}

func (s *Service) mapTripsListItem(item models.TripItemRsp) (result *trips.TripsListItem) {
	switch value := item.(type) {
	case *models.OrderTripItemRsp:
		return &trips.TripsListItem{
			Type:  trips.TripsListItemType_ORDER_TYPE,
			State: value.State,
			Item: &trips.TripsListItem_OrderItem{
				OrderItem: &trips.OrderItem{
					OrderId:     value.OrderID,
					Title:       value.Title,
					Image:       value.Image,
					DisplayDate: value.DisplayDate,
				},
			},
		}
	case *models.RealTripItemRsp:
		return &trips.TripsListItem{
			Type:  trips.TripsListItemType_REAL_TYPE,
			State: value.State,
			Item: &trips.TripsListItem_RealItem{
				RealItem: &trips.RealItem{
					Id:          value.ID,
					Title:       value.Title,
					Image:       value.Image,
					DisplayDate: value.DisplayDate,
				},
			},
		}
	}
	return nil
}

func (s *Service) GetServiceRegisterer() func(*grpc.Server) {
	return func(server *grpc.Server) {
		trips.RegisterTripsServiceV1Server(server, s)
	}
}

func (s *Service) mapAsyncBlocks(ctx context.Context, asyncBlocks []models.AsyncBlockRsp) []*trips.Block {
	blocks := make([]models.BlockRsp, 0, len(asyncBlocks))
	for _, block := range asyncBlocks {
		blocks = append(blocks, models.BlockRsp(block))
	}
	return s.mapBlocks(ctx, blocks)
}

func (s *Service) mapBlocks(ctx context.Context, blocks []models.BlockRsp) []*trips.Block {
	var result []*trips.Block
	for _, block := range blocks {
		switch typedBlock := block.(type) {
		case models.AviaOrders:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_AviaOrders{
						AviaOrders: s.mapAviaOrders(typedBlock),
					},
					Type: trips.BlockType_AVIA_ORDER_BLOCK_TYPE,
				},
			)
		case models.TrainOrders:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_TrainOrders{
						TrainOrders: s.mapTrainOrders(typedBlock),
					},
					Type: trips.BlockType_TRAIN_ORDER_BLOCK_TYPE,
				},
			)
		case models.BusOrders:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_BusOrders{
						BusOrders: s.mapBusOrders(typedBlock),
					},
					Type: trips.BlockType_BUS_ORDER_BLOCK_TYPE,
				},
			)
		case models.HotelOrders:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_HotelOrders{
						HotelOrders: s.mapHotelOrders(typedBlock),
					},
					Type: trips.BlockType_HOTELS_ORDER_BLOCK_TYPE,
				},
			)
		case *models.HotelsCrossSale:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_HotelsCrossSale{
						HotelsCrossSale: s.mapHotelsCrossSale(typedBlock),
					},
					Type: trips.BlockType_HOTELS_CROSS_SALE_BLOCK_TYPE,
				},
			)

		case models.ActivitiesBlock:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_ActivitiesBlock{
						ActivitiesBlock: s.mapActivitiesBlock(typedBlock),
					},
					Type: trips.BlockType_ACTIVITIES_BLOCK_TYPE,
				},
			)

		case *models.RestrictionsBlock:
			result = append(
				result, &trips.Block{
					Data: &trips.Block_RestrictionsBlock{
						RestrictionsBlock: s.mapRestrictionsBlock(typedBlock),
					},
					Type: trips.BlockType_RESTRICTIONS_BLOCK_TYPE,
				},
			)
		case models.NotificationsBlock:
			result = append(
				result,
				&trips.Block{
					Data: &trips.Block_Notifications{
						Notifications: s.mapNotifications(ctx, typedBlock),
					},
					Type: trips.BlockType_NOTIFICATIONS_BLOCK_TYPE,
				},
			)
		case *models.WeatherBlock:
			result = append(
				result,
				&trips.Block{
					Data: &trips.Block_Forecast{
						Forecast: s.mapWeather(typedBlock),
					},
					Type: trips.BlockType_FORECAST_BLOCK_TYPE,
				},
			)
		}
	}
	return result
}

func (s *Service) mapAviaOrders(block models.AviaOrders) *trips.AviaOrders {
	var orders []*trips.AviaOrder
	for _, order := range block.Values {
		var airlines []*rasp.TCarrier
		for _, airline := range order.Companies {
			airlines = append(
				airlines, &rasp.TCarrier{
					Title:       airline.Title,
					SvgLogo:     airline.LogoURL,
					LogoBgColor: airline.Color,
				},
			)
		}

		orders = append(
			orders, &trips.AviaOrder{
				Id:                  order.ID,
				Title:               order.Title,
				DisplayDateForward:  order.DisplayDateForward,
				DisplayDateBackward: order.DisplayDateBackward,
				Pnr:                 order.Pnr,
				Airlines:            airlines,
				RegistrationUrl:     order.RegistrationURL,
				State:               order.State,
				TripOrderState:      order.TripOrderState,
			},
		)
	}
	return &trips.AviaOrders{Values: orders}
}

func (s *Service) mapTrainOrders(block models.TrainOrders) *trips.TrainOrders {
	var orders []*trips.TrainOrder
	for _, order := range block.Values {
		var trains []*trips.TrainOrder_Train
		for _, train := range order.Trains {
			trains = append(
				trains, &trips.TrainOrder_Train{
					Number:      train.Number,
					Description: train.Description,
				},
			)
		}
		orders = append(
			orders, &trips.TrainOrder{
				Id:                           order.ID,
				Title:                        order.Title,
				DisplayDateForward:           order.DisplayDateForward,
				DisplayDateBackward:          order.DisplayDateBackward,
				PrintUrl:                     order.PrintURL,
				Trains:                       trains,
				State:                        order.State,
				HasTransferWithStationChange: order.HasTransferWithStationChange,
				TripOrderState:               order.TripOrderState,
				RefundedTicketsCount:         uint32(order.RefundedTicketsCount),
			},
		)
	}
	return &trips.TrainOrders{Values: orders}
}

func (s *Service) mapHotelOrders(block models.HotelOrders) *trips.HotelOrders {
	var orders []*trips.HotelOrder
	for _, order := range block.Values {
		orders = append(
			orders, s.mapHotelOrder(order),
		)
	}
	return &trips.HotelOrders{Values: orders}
}

func (s *Service) mapHotelOrder(order models.HotelOrder) *trips.HotelOrder {
	return &trips.HotelOrder{
		Id:             order.ID,
		Title:          order.Title,
		Stars:          order.Stars,
		DisplayDates:   order.DisplayDates,
		Address:        order.Address,
		Image:          &commons.Image{Url: order.Image},
		Coordinates:    order.Coordinates,
		State:          order.State,
		DocumentUrl:    order.DocumentURL,
		TripOrderState: order.TripOrderState,
	}
}

func (s *Service) mapHotelsCrossSale(block *models.HotelsCrossSale) *trips.HotelsCrossSale {
	var childrenAges []uint32
	for _, age := range block.ChildrenAges {
		childrenAges = append(childrenAges, uint32(age))
	}

	return &trips.HotelsCrossSale{
		Title:              block.Title,
		CheckInDate:        mapProtoDate(block.CheckInDate),
		CheckOutDate:       mapProtoDate(block.CheckOutDate),
		Adults:             uint32(block.Adults),
		ChildrenAges:       childrenAges,
		SettlementPointKey: block.SettlementPointKey,
		TotalHotelLimit:    uint32(block.TotalHotelLimit),
	}
}

func (s *Service) mapActivitiesBlock(block models.ActivitiesBlock) *trips.ActivitiesBlock {
	resultActivities := make([]*trips.Activity, 0, len(block.Blocks))
	for _, activity := range block.Blocks {
		var resultActivity *trips.Activity
		switch value := activity.(type) {
		case *models.AfishaEventInfo:
			resultActivity = &trips.Activity{
				Type: trips.ActivityType_AFISHA,
				Payload: &trips.Activity_AfishaEventInfo{
					AfishaEventInfo: &trips.AfishaEventInfo{
						Name:     value.Name,
						MinPrice: mapPrice(value.MinPrice),
						Type:     value.Type,
						DateTime: value.DateTime,
						DateText: value.DateText,
						ImageUrl: value.ImageURL,
						EventUrl: value.EventURL,
						Tags:     value.Tags,
					},
				},
			}
		case *models.IziTravelTourInfo:
			resultActivity = &trips.Activity{
				Type: trips.ActivityType_IZI_TRAVEL,
				Payload: &trips.Activity_IziTravelTourInfo{
					IziTravelTourInfo: &trips.IziTravelTourInfo{
						Name:     value.Name,
						ImageUrl: value.ImageURL,
						TourUrl:  value.TourURL,
						Type:     value.Type,
						Category: value.Category,
						Duration: int32(value.Duration),
					},
				},
			}
		}
		resultActivities = append(resultActivities, resultActivity)
	}
	return &trips.ActivitiesBlock{
		IsLoaded:   len(resultActivities) > 0,
		Activities: resultActivities,
	}
}

func (s *Service) mapRestrictionsBlock(block *models.RestrictionsBlock) *trips.RestrictionsBlock {
	return &trips.RestrictionsBlock{
		FromCountryTitle: block.FromCountryTitle,
		ToCountryTitle:   block.ToCountryTitle,
		FromGeoId:        int32(block.FromGeoID),
		ToGeoId:          int32(block.ToGeoID),
		FromPointKey:     block.FromPointKey,
		ToPointKey:       block.ToPointKey,
	}
}

func (s *Service) mapBusOrders(block models.BusOrders) *trips.BusOrders {
	var orders []*trips.BusOrder
	for _, order := range block.Values {
		orders = append(
			orders, &trips.BusOrder{
				Id:                   order.ID,
				Title:                order.Title,
				Description:          order.Description,
				DisplayDateForward:   order.DisplayDateForward,
				DownloadBlankToken:   order.DownloadBlankToken,
				CarrierName:          order.CarrierName,
				State:                order.State,
				TripOrderState:       order.TripOrderState,
				RefundedTicketsCount: uint32(order.RefundedTicketsCount),
			},
		)
	}
	return &trips.BusOrders{Values: orders}
}

func (s *Service) mapNotifications(ctx context.Context, sourceNotifications models.NotificationsBlock) *trips.NotificationsBlock {
	notifications := make([]*trips.Notification, 0, len(sourceNotifications.Blocks))
	for _, n := range sourceNotifications.Blocks {
		mappedNotification, err := s.mapNotification(n)
		if err != nil {
			ctxlog.Error(ctx, s.logger, "failed to map notification for trip", log.Error(err))
			continue
		}
		notifications = append(notifications, mappedNotification)
	}
	return &trips.NotificationsBlock{Notifications: notifications}
}

func (s *Service) mapNotification(notification models.Notification) (*trips.Notification, error) {
	notificationType := notification.Type()
	switch notificationType {
	case models.NotificationTypeAviaOnlineCheckIn:
		typedNotification := notification.(*models.AviaOnlineCheckInNotification)
		return &trips.Notification{
			Type: trips.NotificationType_NOTIFICATION_TYPE_AVIA_ONLINE_CHECK_IN,
			Data: &trips.Notification_AviaOnlineCheckIn{AviaOnlineCheckIn: &trips.AviaOnlineCheckIn{
				OrderId: typedNotification.OrderID,
				Airline: &rasp.TCarrier{
					Title:       typedNotification.Airline.Title,
					SvgLogo:     typedNotification.Airline.LogoURL,
					LogoBgColor: typedNotification.Airline.Color,
				},
				FlightNumber:    typedNotification.FlightNumber,
				FlightTitle:     typedNotification.FlightTitle,
				OfflineCheckIn:  s.mapOfflineCheckIn(typedNotification.AviaOfflineCheckIn),
				RegistrationUrl: typedNotification.RegistrationURL,
				Pnr:             typedNotification.Pnr,
				UpdatedAt:       typedNotification.UpdatedAt,
			}},
		}, nil
	case models.NotificationTypeHotelDeferredPayment:
		typedNotification := notification.(*models.HotelDeferredPaymentNotification)
		return &trips.Notification{
			Type: trips.NotificationType_NOTIFICATION_TYPE_HOTEL_DEFERRED_PAYMENT,
			Data: &trips.Notification_HotelDeferredPayment{HotelDeferredPayment: &trips.HotelDeferredPayment{
				Order: s.mapHotelOrder(typedNotification.Order),
			}},
		}, nil

	default:
		return nil, fmt.Errorf("unsupported kind of notification: %d", notificationType)
	}
}

func (s *Service) mapOfflineCheckIn(offlineCheckIn models.AviaOfflineCheckIn) *trips.AviaOfflineCheckIn {
	if offlineCheckIn.CheckInCounters != "" || offlineCheckIn.Gate != "" {
		return &trips.AviaOfflineCheckIn{
			CheckInCounters: offlineCheckIn.CheckInCounters,
			Gate:            offlineCheckIn.Gate,
		}
	}
	return nil
}

func (s *Service) mapWeather(block *models.WeatherBlock) *trips.ForecastBlock {
	items := make([]*trips.ForecastItem, 0, len(block.Forecast))
	for _, weatherItem := range block.Forecast {
		items = append(items, &trips.ForecastItem{
			ImageUrl:    weatherItem.ImageURL,
			Title:       weatherItem.Title,
			Description: weatherItem.Description,
			Url:         weatherItem.URL,
			ItemType:    weatherItem.ItemType,
		})
	}
	return &trips.ForecastBlock{Items: items}
}

func mapPrice(price *models.Price) *trips.AfishaEventInfo_Price {
	if price == nil {
		return nil
	}
	return &trips.AfishaEventInfo_Price{
		Value:    float32(price.Value),
		Currency: price.Currency,
	}
}

func mapProtoDate(date time.Time) *travelcommonsproto.TDate {
	return &travelcommonsproto.TDate{
		Year:  int32(date.Year()),
		Month: int32(date.Month()),
		Day:   int32(date.Day()),
	}
}
