package http

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/go-chi/chi/v5"
	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	api "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips"
	"a.yandex-team.ru/travel/library/go/httputil"
)

type Handler struct {
	logger     log.Logger
	provider   *api.Provider
	tripMapper TripMapper
}

func NewHandler(logger log.Logger, provider *api.Provider) *Handler {
	return &Handler{logger: logger, provider: provider}
}

func (h *Handler) GetRouteBuilder() func(r chi.Router) {
	return func(r chi.Router) {
		r.Get("/external-api/v1/get-active-trips", h.GetActiveTrips)
	}
}

func (h *Handler) GetActiveTrips(w http.ResponseWriter, r *http.Request) {
	span, ctx := opentracing.StartSpanFromContext(r.Context(), "components.api.trips.http.Handler.GetActiveTrips")
	defer span.Finish()

	var form GetActiveTripsForm
	if err := form.Parse(r); err != nil {
		httputil.HandleError(err, http.StatusBadRequest, w)
		return
	}

	activeTrips, err := h.provider.GetActiveTrips(ctx, form.PassportID, uint(form.Limit), form.GeoID)
	if err != nil {
		httputil.HandleError(
			fmt.Errorf("failed to handle request: %w", err),
			http.StatusInternalServerError,
			w,
		)
		return
	}

	response := h.tripMapper.MapTripsList(ctx, activeTrips)
	h.writeResponse(ctx, w, response)
}

func (h *Handler) writeResponse(ctx context.Context, w http.ResponseWriter, response GetActiveTripsResponse) {
	span, _ := opentracing.StartSpanFromContext(ctx, "components.api.trips.http.Handler.EncodeResponse")
	defer span.Finish()

	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(http.StatusOK)
	if err := json.NewEncoder(w).Encode(response); err != nil {
		httputil.HandleError(
			fmt.Errorf("failed to encode json: %w", err),
			http.StatusInternalServerError,
			w,
		)
		return
	}
}
