package http

import (
	"context"
	"fmt"

	"github.com/opentracing/opentracing-go"

	tripsapi "a.yandex-team.ru/travel/komod/trips/api/trips/v1"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
)

type TripMapper struct{}

func (m *TripMapper) MapTripsList(ctx context.Context, items []apimodels.TripItemRsp) GetActiveTripsResponse {
	span, _ := opentracing.StartSpanFromContext(ctx, "components.api.trips.http.TripMapper.MapTripsList")
	defer span.Finish()

	trips := make([]*TripListItem, 0, len(items))
	for _, item := range items {
		trips = append(trips, m.MapTripsListItem(item))
	}
	return GetActiveTripsResponse{
		Trips: trips,
	}
}

func (m *TripMapper) MapTripsListItem(item apimodels.TripItemRsp) *TripListItem {
	switch value := item.(type) {
	case *apimodels.OrderTripItemRsp:
		return &TripListItem{
			Type: OrderItem,
			Item: &TripListItemOrder{
				OrderID:     value.OrderID,
				Title:       value.Title,
				Image:       value.Image,
				DisplayDate: value.DisplayDate,
			},
			State: MapTripState(value.State),
		}
	case *apimodels.RealTripItemRsp:
		return &TripListItem{
			Type: RealItem,
			Item: &TripListItemReal{
				ID:          value.ID,
				Title:       value.Title,
				Image:       value.Image,
				DisplayDate: value.DisplayDate,
			},
			State: MapTripState(value.State),
		}
	default:
		panic(fmt.Sprintf("unexpected TripItem type: %+v", value))
	}
}

func MapTripState(state tripsapi.TripState) TripListItemState {
	switch state {
	case tripsapi.TripState_TRIP_STATE_CONFIRMED:
		return TripStateConfirmed
	case tripsapi.TripState_TRIP_STATE_CANCELLED:
		return TripStateCancelled
	default:
		panic(fmt.Sprintf("unexpected TripState type: %+v", state))
	}
}
