package trips

import (
	"fmt"
	"sort"

	"github.com/jonboulle/clockwork"

	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
)

type Page struct {
	trips     tripsmodels.Trips
	nextToken *apimodels.ContinuationToken
}

type StartPageBuilder struct {
	clock clockwork.Clock
}

func NewStartPageBuilder(clock clockwork.Clock) *StartPageBuilder {
	return &StartPageBuilder{clock: clock}
}

func (p StartPageBuilder) BuildPage(sourceTripsList tripsmodels.Trips, token *apimodels.ContinuationToken, limit uint) (*Page, error) {
	tripsList := make([]*tripsmodels.Trip, len(sourceTripsList))
	copy(tripsList, sourceTripsList)

	if token.NextTripID == nil {
		tripsList = p.filterTripsListByNow(tripsList, token)
	}
	tripsList = p.getSortedTripsList(tripsList, token)

	var err error
	if token.NextTripID != nil {
		tripsList, err = p.filterTripsListByNextTripID(tripsList, *token.NextTripID)
		if err != nil {
			return nil, err
		}
	}

	var nextToken *apimodels.ContinuationToken
	if len(tripsList) > int(limit) {
		nextToken = &apimodels.ContinuationToken{
			TripsType:  token.TripsType,
			NextTripID: &tripsList[limit].ID,
		}
		tripsList = tripsList[:limit]
	}
	return &Page{
		trips:     tripsList,
		nextToken: nextToken,
	}, nil
}

func (p StartPageBuilder) getSortedTripsList(tripsList tripsmodels.Trips, token *apimodels.ContinuationToken) []*tripsmodels.Trip {
	result := make([]*tripsmodels.Trip, len(tripsList))
	copy(result, tripsList)

	switch token.TripsType {
	case apimodels.ActiveTrips:
		sort.Stable(byStart(result))
	case apimodels.PastTrips:
		sort.Stable(sort.Reverse(byStart(result)))
	}
	return result
}

func (p StartPageBuilder) filterTripsListByNow(tripsList tripsmodels.Trips, token *apimodels.ContinuationToken) (result tripsmodels.Trips) {
	var isGoodTrip func(*tripsmodels.Trip) bool
	var now = p.clock.Now()

	isActive := func(trip *tripsmodels.Trip) bool {
		return !trip.Cancelled() && trip.EndTime().After(now)
	}
	isDisabled := func(trip *tripsmodels.Trip) bool {
		return !isActive(trip)
	}

	switch token.TripsType {
	case apimodels.ActiveTrips:
		isGoodTrip = isActive
	case apimodels.PastTrips:
		isGoodTrip = isDisabled
	}

	for _, trip := range tripsList {
		if isGoodTrip(trip) {
			result = append(result, trip)
		}
	}
	return result
}

func (p StartPageBuilder) filterTripsListByNextTripID(tripsList tripsmodels.Trips, nextTripID string) (tripsmodels.Trips, error) {
	var lowerBound *int
	for i, trip := range tripsList {
		if trip.ID == nextTripID {
			lowerBound = &i
			break
		}
	}
	if lowerBound == nil {
		return nil, fmt.Errorf("unable to find next trip %s", nextTripID)
	}

	return tripsList[*lowerBound:], nil
}
