package trips

import (
	"context"
	"errors"
	"fmt"
	"net/url"
	"path"
	"sort"
	"strings"
	"time"

	"github.com/go-openapi/strfmt"
	opentracing "github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/async"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/displaytime"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/weather"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/extractors"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/pgclient"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/komod/trips/internal/services"
	"a.yandex-team.ru/travel/komod/trips/internal/sharedflights"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	"a.yandex-team.ru/travel/komod/trips/internal/trips"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/errutil"
	"a.yandex-team.ru/travel/library/go/geobase"
	"a.yandex-team.ru/travel/library/go/strutil"
	"a.yandex-team.ru/travel/library/go/syncutil"
	pullnotifications "a.yandex-team.ru/travel/notifier/api/pull_notifications/v1"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

const (
	providerName  = "api.trips.Provider"
	airlineCodeDP = 9144
	airlineCodeFV = 8565
	airlineCodeHZ = 50
	airlineCodeSU = 26
)

var AirlinesWithAeroflotFlightCheckIn = containers.SetOf(
	airlineCodeDP, airlineCodeFV, airlineCodeHZ, // RASPTICKETS-23057
)

type Provider struct {
	config                ProviderConfig
	logger                log.Logger
	pgClient              *pgclient.Client
	tripsStorage          trips.Storage
	orderClient           orders.Client
	orderInfoExtractor    *extractors.OrderInfoExtractor
	pageBuilder           *StartPageBuilder
	geoBase               geobase.Geobase
	spanHelper            *span.Helper
	mapper                *TripsMapper
	asyncBlockProvider    *async.BlockProvider
	restrictionsProvider  *RestrictionsProvider
	crossSaleProvider     *CrossSaleProvider
	notificationsProvider notificationsProvider
	weatherProvider       *weather.Provider

	unprocessedEntityService      services.UnprocessedOrders
	flightInfoProvider            flightInfoProvider
	references                    references.References
	stationIDToSettlementIDMapper stationIDToSettlementIDMapper
	raspMediaURL                  string
}

type notificationsProvider interface {
	GetNotificationsByOrderIDs(ctx context.Context, orderIDs []string) (*pullnotifications.GetPullNotificationsByOrdersIdsRspV1, error)
}

type flightInfoProvider interface {
	GetFlightInfo(ctx context.Context, flightNumber string, departureDate time.Time) (sharedflights.Flight, error)
}

type stationIDToSettlementIDMapper interface {
	Map(stationID int32) (int32, bool)
}

func NewTripsProvider(
	cfg ProviderConfig,
	raspMediaURL string,
	logger log.Logger,
	pgClient *pgclient.Client,
	tripsStorage trips.Storage,
	orderClient orders.Client,
	orderInfoExtractor *extractors.OrderInfoExtractor,
	pageBuilder *StartPageBuilder,
	geoBase geobase.Geobase,
	mapper *TripsMapper,
	asyncBlockProvider *async.BlockProvider,
	unprocessedEntityService services.UnprocessedOrders,
	restrictionsProvider *RestrictionsProvider,
	crossSaleProvider *CrossSaleProvider,
	notificationsProvider notificationsProvider,
	weatherProvider *weather.Provider,
	flightInfoProvider flightInfoProvider,
	references references.References,
	stationIDToSettlementIDMapper stationIDToSettlementIDMapper,
) *Provider {
	return &Provider{
		config:                        cfg,
		logger:                        logger.WithName(providerName),
		pgClient:                      pgClient,
		tripsStorage:                  tripsStorage,
		orderClient:                   orderClient,
		orderInfoExtractor:            orderInfoExtractor,
		pageBuilder:                   pageBuilder,
		geoBase:                       geoBase,
		mapper:                        mapper,
		asyncBlockProvider:            asyncBlockProvider,
		unprocessedEntityService:      unprocessedEntityService,
		restrictionsProvider:          restrictionsProvider,
		crossSaleProvider:             crossSaleProvider,
		notificationsProvider:         notificationsProvider,
		weatherProvider:               weatherProvider,
		flightInfoProvider:            flightInfoProvider,
		references:                    references,
		stationIDToSettlementIDMapper: stationIDToSettlementIDMapper,
		raspMediaURL:                  raspMediaURL,
	}
}

func (p *Provider) GetActiveTrips(ctx context.Context, passportID string, limit uint, geoID int) (
	response []apimodels.TripItemRsp,
	err error,
) {
	var funcName = fmt.Sprintf("%s.GetActiveTrips", providerName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	tripsList, err := p.tripsStorage.GetSession().GetTrips(ctx, passportID)
	if err != nil {
		return nil, err
	}

	page, err := p.pageBuilder.BuildPage(tripsList, GenerateStartToken(apimodels.ActiveTrips), limit)
	if err != nil {
		return nil, err
	}
	userNow := p.getUserNow(ctx, geoID)
	return p.mapper.MapTrips(page.trips, userNow, consts.TripInActionMenu), nil
}

func (p *Provider) GetTrips(
	ctx context.Context,
	passportID string,
	pastLimit uint,
	geoID int,
) (response *apimodels.TripsRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetTrips", providerName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	tripsList, err := p.tripsStorage.GetSession().GetTrips(ctx, passportID)
	if err != nil {
		return nil, err
	}

	orderIDs := extractOrderIDsFromTrips(tripsList...)
	unprocessedOrders := p.getUnprocessedOrders(ctx, passportID, orderIDs...)
	fakeTrips := p.makeFakeTripsFromOrders(ctx, passportID, unprocessedOrders...)

	activePage, err := p.pageBuilder.BuildPage(
		tripsList,
		GenerateStartToken(apimodels.ActiveTrips),
		uint(len(tripsList)),
	)
	if err != nil {
		return nil, err
	}

	pastPage, err := p.pageBuilder.BuildPage(tripsList, GenerateStartToken(apimodels.PastTrips), pastLimit)
	if err != nil {
		return nil, err
	}
	pastPage.trips = append(pastPage.trips, fakeTrips...)

	userNow := p.getUserNow(ctx, geoID)
	return &apimodels.TripsRsp{
		Active: p.mapTripListPage(activePage, userNow, getImageAliasByToken(apimodels.ActiveTrips)),
		Past:   p.mapTripListPage(pastPage, userNow, getImageAliasByToken(apimodels.PastTrips)),
	}, nil
}

func extractOrderIDsFromTrips(tripsList ...*tripsmodels.Trip) []orders.ID {
	var orderIDs []orders.ID
	for _, trip := range tripsList {
		orderIDs = append(orderIDs, trip.GetOrderIDs()...)
	}
	return orderIDs
}

func (p *Provider) GetMoreTrips(
	ctx context.Context,
	passportID string,
	tokenValue string,
	limit uint,
	geoID int,
) (response *apimodels.PaginatedTripsListRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetMoreTrips", providerName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	token, err := LoadToken(tokenValue)
	if err != nil {
		return nil, err
	}

	tripsList, err := p.tripsStorage.GetSession().GetTrips(ctx, passportID)
	if err != nil {
		return nil, err
	}

	page, err := p.pageBuilder.BuildPage(tripsList, token, limit)
	if err != nil {
		return nil, err
	}

	userNow := p.getUserNow(ctx, geoID)
	return p.mapTripListPage(page, userNow, getImageAliasByToken(token.TripsType)), nil
}

func getImageAliasByToken(tripsType apimodels.TripsType) consts.ImageAlias {
	switch tripsType {
	case apimodels.ActiveTrips:
		return consts.LargeActiveTrip
	case apimodels.PastTrips:
		return consts.PastTrip
	default:
		panic(fmt.Sprintf("unexpected trip type '%s'", tripsType))
	}
}

func (p *Provider) GetTrip(
	ctx context.Context,
	passportID string,
	tripID string,
	geoID int,
) (response *apimodels.TripRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetTrip", providerName)
	defer errutil.Wrap(&err, funcName)
	ctx = ctxlog.WithFields(ctx, log.String("passportID", passportID), log.String("tripID", tripID))

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	ctx, cancel := context.WithCancel(ctx)
	defer cancel()

	trip, err := p.tripsStorage.GetSession().GetTrip(ctx, tripID)
	if err != nil {
		return nil, err
	}

	if trip == nil {
		return nil, ErrTripNotFound
	}
	if trip.PassportID != passportID {
		return nil, ErrTripIsForbidden
	}

	var ws syncutil.WaitGroup
	var weatherBlock apimodels.BlockRsp
	var ordersList []orders.Order
	ws.Go(func() {
		weatherBlock = p.getWeatherBlock(ctx, trip)
	})

	ordersList, err = p.orderClient.GetOrdersByIDs(ctx, trip.GetOrderIDs()...)
	if err != nil {
		return nil, err
	}

	userNow := p.getUserNow(ctx, geoID)
	blocks := []apimodels.BlockRsp{p.getNotificationsBlock(ctx, ordersList, userNow)}
	blocks = append(blocks, p.mapper.MapBlocksRspByVertical(ctx, ordersList, userNow)...)
	blocks = p.addActivitiesBlock(trip, blocks)
	blocks = p.addCrossSaleBlock(trip, blocks)
	blocks = p.addRestrictionsBlock(trip, blocks)

	ws.Wait()
	if weatherBlock != nil {
		blocks = append(blocks, weatherBlock)
	}

	visits := p.mapper.extractActiveOrAllVisitsFromTrip(trip)

	tripStart := span.GetStartTime(trip.GetSpans())
	tripEnd := span.GetEndTime(trip.GetSpans())

	return &apimodels.TripRsp{
		ID:    trip.ID,
		Title: extractTripTitle(visits),
		State: trip.State(),
		Image: p.mapper.extractTripImage(visits, consts.TripHead),
		DisplayDate: displaytime.NewBuilder().BuildDatesPair(
			displaytime.NewDatesPairArgs(
				userNow, tripStart, tripEnd,
			).SetUseRelative().SetUseYear(),
		),
		BeginDate: tripStart,
		EndDate:   tripEnd,
		Blocks:    blocks,
	}, nil
}

func (p *Provider) addCrossSaleBlock(trip *tripsmodels.Trip, blocks []apimodels.BlockRsp) []apimodels.BlockRsp {
	if !p.config.CrossSaleBlockEnabled {
		return blocks
	}

	crossSaleBlock := p.crossSaleProvider.MakeBlock(trip)
	if crossSaleBlock != nil {
		blocks = append(blocks, crossSaleBlock)
	}
	return blocks
}

func (p *Provider) addRestrictionsBlock(trip *tripsmodels.Trip, blocks []apimodels.BlockRsp) []apimodels.BlockRsp {
	if !p.config.RestrictionsBlockEnabled {
		return blocks
	}

	restrictionsBlock := p.restrictionsProvider.MakeBlock(trip)
	if restrictionsBlock != nil {
		blocks = append([]apimodels.BlockRsp{restrictionsBlock}, blocks...)
	}
	return blocks
}

func (p *Provider) getWeatherBlock(ctx context.Context, trip *tripsmodels.Trip) apimodels.BlockRsp {
	if !p.config.WeatherEnabled {
		return nil
	}
	return p.weatherProvider.MakeBlock(ctx, trip)
}

func (p *Provider) addActivitiesBlock(trip *tripsmodels.Trip, blocks []apimodels.BlockRsp) []apimodels.BlockRsp {
	if !p.config.ActivitiesBlockEnabled {
		return blocks
	}

	if p.asyncBlockProvider.BlockIsAvailable(async.ActivitiesBlock, trip) {
		blocks = append(blocks, p.asyncBlockProvider.GetEmptyBlockForType(async.ActivitiesBlock))
	}
	return blocks
}

func (p *Provider) GetTripIDByOrderID(
	ctx context.Context,
	passportID string,
	orderID string,
) (tripID string, err error) {
	var funcName = fmt.Sprintf("%s.GetTripItemByOrder", providerName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	tripsList, err := p.tripsStorage.GetSession().GetTrips(ctx, passportID)
	if err != nil {
		return "", err
	}

	trip := p.filterTripByOrderID(tripsList, orders.ID(orderID))

	if trip == nil {
		return "", ErrTripNotFound
	}
	return trip.ID, nil
}

func (p *Provider) GetTripAsyncBlocks(
	ctx context.Context,
	passportID string,
	tripID string,
	blockTypes []async.BlockType,
) (_ []apimodels.AsyncBlockRsp, err error) {
	var funcName = fmt.Sprintf("%s.GetTripAsyncBlocks", providerName)
	defer errutil.Wrap(&err, funcName)

	tracingSpan, ctx := opentracing.StartSpanFromContext(ctx, funcName)
	defer tracingSpan.Finish()

	trip, err := p.tripsStorage.GetSession().GetTrip(ctx, tripID)
	if err != nil {
		return nil, err
	}

	if trip == nil {
		return nil, ErrTripNotFound
	}
	if trip.PassportID != passportID {
		return nil, ErrTripIsForbidden
	}

	return p.asyncBlockProvider.GetBlocks(ctx, blockTypes, trip)
}

func (p *Provider) filterTripByOrderID(tripsList tripsmodels.Trips, queryOrderID orders.ID) *tripsmodels.Trip {
	for _, trip := range tripsList {
		for _, orderID := range trip.GetOrderIDs() {
			if orderID == queryOrderID {
				return trip
			}
		}
	}
	return nil
}

func (p *Provider) makeFakeTripsFromOrders(ctx context.Context, passportID string, orders ...orders.Order) tripsmodels.Trips {
	var tripsList tripsmodels.Trips
	for _, order := range orders {
		fakeTrip := tripsmodels.NewTrip("", passportID)
		orderInfo, err := p.orderInfoExtractor.Extract(order)
		if err != nil {
			ctxlog.Error(
				ctx,
				p.logger,
				"failed to build fake trip from order",
				log.String("orderID", order.ID().String()),
			)
			continue
		}
		fakeTrip.UpsertOrder(orderInfo)
		tripsList = append(tripsList, fakeTrip)
	}
	return tripsList
}

func (p *Provider) getUserNow(ctx context.Context, geoID int) time.Time {
	location, err := p.geoBase.GetLocationByID(geoID)
	if err != nil {
		ctxlog.Error(
			ctx,
			p.logger,
			"unable to find location",
			log.String("method", "getUserNow"),
			log.Int("geoId", geoID),
			log.Error(err),
		)
		return time.Now().In(consts.MskLocation)
	}
	return time.Now().In(location)
}

func (p *Provider) mapTripListPage(
	page *Page,
	userNow time.Time,
	alias consts.ImageAlias,
) *apimodels.PaginatedTripsListRsp {
	var tokenValue string
	if page.nextToken != nil {
		tokenValue = DumpToken(*page.nextToken)
	}

	return &apimodels.PaginatedTripsListRsp{
		Trips:             p.mapper.MapTrips(page.trips, userNow, alias),
		ContinuationToken: tokenValue,
	}
}

func (p *Provider) enqueueUnprocessedOrders(ctx context.Context, ordersList []orders.Order) {
	unprocessedPuttingCtx, cancel := context.WithTimeout(context.Background(), 10*time.Second)
	defer cancel()
	for _, order := range ordersList {
		const defaultRetriesCount = 3
		err := p.unprocessedEntityService.PutOrderID(unprocessedPuttingCtx, order.ID(), defaultRetriesCount)
		if err != nil {
			ctxlog.Error(
				ctx,
				p.logger,
				"failed to enqueue unprocessed order id",
				log.String("orderID", order.ID().String()),
				log.Error(err),
			)
		}
	}
}

func (p *Provider) getUnprocessedOrders(ctx context.Context, passportID string, orderIDs ...orders.ID) []orders.Order {
	page, nextPageToken, err := p.orderClient.GetUserOrdersWithoutExcluded(ctx, passportID, orderIDs...)
	if err != nil {
		ctxlog.Error(
			ctx,
			p.logger,
			"failed to get first page of unprocessed orders",
			log.Error(err),
		)
		return nil
	}

	go p.enqueueUnprocessedOrders(ctx, page)
	go func(token string) {
		requestCtx, cancelFunc := context.WithTimeout(context.Background(), 60*time.Second)
		defer cancelFunc()
		var err error
		var nextPage []orders.Order
		for token != "" {
			nextPage, token, err = p.orderClient.GetUserOrdersWithoutExcludedNextPage(requestCtx, token)
			if err != nil {
				ctxlog.Error(
					ctx,
					p.logger,
					"failed to get next page of unprocessed orders",
					log.Error(err),
				)
				return
			} else {
				go p.enqueueUnprocessedOrders(ctx, nextPage)
			}
		}
	}(nextPageToken)

	return page
}

func (p *Provider) getNotificationsBlock(ctx context.Context, ordersList []orders.Order, userNow time.Time) apimodels.NotificationsBlock {
	orderIDs := make([]string, 0, len(ordersList))
	orderByID := make(map[orders.ID]orders.Order, len(ordersList))
	for _, o := range ordersList {
		if o.Cancelled() || o.Refunded() {
			continue
		}
		orderIDs = append(orderIDs, o.ID().String())
		orderByID[o.ID()] = o
	}

	notificationsResponse, err := p.notificationsProvider.GetNotificationsByOrderIDs(ctx, orderIDs)
	if err != nil {
		ctxlog.Error(ctx, p.logger, "failed to get notifications", log.Error(err))
		return apimodels.NotificationsBlock{Blocks: []apimodels.Notification{}}
	}
	notifications := p.buildNotifications(ctx, notificationsResponse.Notifications, orderByID, ordersList, userNow)
	return apimodels.NotificationsBlock{Blocks: notifications}
}

func (p *Provider) buildNotifications(
	ctx context.Context,
	notifications []*pullnotifications.PullNotification,
	orderByID map[orders.ID]orders.Order,
	ordersList []orders.Order,
	userNow time.Time,
) []apimodels.Notification {
	resultsChan := make(chan apimodels.Notification, len(notifications))
	wg := syncutil.WaitGroup{}
	priority := 0
	for _, n := range notifications {
		localNotification := n
		notificationPriority := priority
		wg.Go(func() {
			var order orders.Order
			if orderID := localNotification.GetOrderId(); orderID != "" {
				order = orderByID[orders.ID(orderID)]
			}
			p.mapNotification(ctx, localNotification, resultsChan, order, notificationPriority)
		})
		priority++
	}
	wg.Wait()
	close(resultsChan)
	results := make([]apimodels.Notification, 0)
	for n := range resultsChan {
		results = append(results, n)
	}
	for _, order := range ordersList {
		if order.Cancelled() || order.Refunded() {
			continue
		}
		if notification, ok := p.buildNotificationFromOrder(order, priority, userNow); ok {
			results = append(results, notification)
			priority++
		}
	}
	sortNotifications(results)
	return results
}

func sortNotifications(results []apimodels.Notification) {
	sort.SliceStable(results, func(i, j int) bool {
		return results[i].Priority() < results[j].Priority()
	})
}

func (p *Provider) mapNotification(
	ctx context.Context,
	notification *pullnotifications.PullNotification,
	resultsChan chan<- apimodels.Notification,
	order orders.Order,
	priority int,
) {
	switch notification.Type {
	case pullnotifications.PullNotificationType_PULL_NOTIFICATION_TYPE_ONLINE_REGISTRATION:
		result, err := p.mapAviaOnlineCheckInNotification(ctx, notification, order.(*orders.AviaOrder), priority)
		if err != nil {
			ctxlog.Error(ctx, p.logger, "failed to build online registration notification", log.Error(err))
		} else {
			resultsChan <- result
		}
	}
}

func (p *Provider) mapAviaOnlineCheckInNotification(
	ctx context.Context,
	notification *pullnotifications.PullNotification,
	order *orders.AviaOrder,
	priority int,
) (*apimodels.AviaOnlineCheckInNotification, error) {
	notificationData := notification.GetData().GetOnlineRegistration()
	rawDepartureDate, err := time.Parse(strfmt.ISO8601TimeWithReducedPrecisionLocaltime, notificationData.LocalDepartureTime)
	if err != nil {
		return nil, fmt.Errorf("invalid departure date format: %s: %w", notificationData.LocalDepartureTime, err)
	}
	flightNumber := notificationData.OperationFlightNumber
	flightInfo, err := p.flightInfoProvider.GetFlightInfo(ctx, flightNumber, rawDepartureDate)
	if err != nil && !errors.Is(err, sharedflights.ErrFlightNotFound) && !errors.Is(err, context.DeadlineExceeded) && !errors.Is(err, context.Canceled) {
		p.logger.Warn("failed to fetch flight info for avia online check-in notification", log.Error(err))
		return nil, fmt.Errorf("failed to fetch flight info: %w", err)
	}
	updatedAt := ""
	if err == nil {
		flightInfo = p.selectSegment(flightInfo, notificationData)
		updatedAt, err = p.buildFlightStatusUpdatedAt(flightInfo.Status)
		if err != nil {
			p.logger.Warn("failed to fetch flight info for avia online check-in notification", log.Error(err))
			return nil, err
		}
	}
	airline, err := p.parseAirline(flightNumber, flightInfo)
	if err != nil {
		return nil, err
	}

	return apimodels.NewAviaOnlineCheckInNotification(
		notification.OrderId,
		p.buildAirline(airline),
		flightNumber,
		p.buildFlightTitle(notificationData),
		apimodels.AviaOfflineCheckIn{
			CheckInCounters: flightInfo.Status.CheckInDesks,
			Gate:            flightInfo.Status.DepartureGate,
		},
		buildRegistrationURL(p.references.Carriers(), airline),
		order.PNR,
		updatedAt,
		priority,
	), nil
}

func (p *Provider) parseAirline(flightNumber string, flightInfo sharedflights.Flight) (*rasp.TCarrier, error) {
	flightNumberParts := strings.Split(flightNumber, " ")
	airlineCode := ""
	if len(flightNumberParts) > 0 {
		airlineCode = flightNumberParts[0]
	}
	if airline, ok := p.references.Carriers().GetByCode(airlineCode); ok {
		return airline, nil
	}
	if airline, ok := p.references.Carriers().Get(flightInfo.AirlineID); flightInfo.AirlineID != 0 && ok {
		return airline, nil
	}
	return nil, fmt.Errorf("failed to parse airline from flight number %s", flightNumber)
}

func (p *Provider) buildFlightTitle(notification *pullnotifications.OnlineRegistration) string {
	departure, err := p.getTitlePart(notification.DepartureStationId)
	if err != nil {
		p.logger.Error("failed to build title", log.Int32("stationID", notification.DepartureStationId))
		return ""
	}

	arrival, err := p.getTitlePart(notification.ArrivalStationId)
	if err != nil {
		p.logger.Error("failed to build title", log.Int32("stationID", notification.DepartureStationId))
		return ""
	}
	return fmt.Sprintf("%s — %s", departure, arrival)
}

func (p *Provider) getTitlePart(stationID int32) (string, error) {
	settlementID, ok := p.stationIDToSettlementIDMapper.Map(stationID)
	if ok {
		settlement, found := p.references.Settlements().Get(int(settlementID))
		if found {
			return strutil.Coalesce(settlement.GetTitle().GetRu().GetNominative(), settlement.GetTitleDefault()), nil
		}
	}
	station, found := p.references.Stations().Get(int(stationID))
	if !found {
		return "", fmt.Errorf("unknown stationID: %d", stationID)
	}
	return strutil.Coalesce(station.GetTitle().GetRu(), station.GetTitleDefault()), nil
}

func buildRegistrationURL(repo *references.CarrierRepository, airline *rasp.TCarrier) string {
	var airlineID int
	if AirlinesWithAeroflotFlightCheckIn.Contains(int(airline.Id)) {
		airlineID = airlineCodeSU
	} else {
		airlineID = int(airline.Id)
	}

	airlineForURL, ok := repo.Get(airlineID)
	if !ok {
		airlineForURL = airline
	}

	return strutil.Coalesce(
		airlineForURL.RegistrationUrl,
		airlineForURL.RegistrationUrlRu,
		airlineForURL.RegistrationUrlEn,
	)
}

func (p *Provider) selectSegment(flight sharedflights.Flight, notification *pullnotifications.OnlineRegistration) sharedflights.Flight {
	for _, s := range flight.Segments {
		// in case of multi-segment flights
		if int32(s.AirportFromID) == notification.DepartureStationId && int32(s.AirportToID) == notification.ArrivalStationId {
			return *s
		}
	}
	return flight
}

func (p *Provider) buildAirline(airline *rasp.TCarrier) apimodels.Company {
	parsedRaspMediaURL, _ := url.Parse(p.raspMediaURL)
	parsedRaspMediaURL.Path = path.Join(parsedRaspMediaURL.Path, airline.SvgLogo)
	return apimodels.Company{
		Title:   airline.Title,
		LogoURL: parsedRaspMediaURL.String(),
		Color:   airline.LogoBgColor,
	}
}

func (p *Provider) buildNotificationFromOrder(order orders.Order, priority int, userNow time.Time) (apimodels.Notification, bool) {
	if hotelOrder, ok := order.(*orders.HotelOrder); ok && hotelOrder.HasDeferredPayment {
		return apimodels.NewHotelDeferredPaymentNotification(p.mapper.mapHotelOrder(hotelOrder, userNow), priority), true
	}
	return nil, false
}

func (p *Provider) buildFlightStatusUpdatedAt(status sharedflights.Status) (string, error) {
	updatedAt, err := time.Parse(strfmt.ISO8601TimeUniversalSortableDateTimePattern, status.UpdatedAtUTC)
	if err != nil {
		return "", fmt.Errorf("failed to parse flight info updatedAt: %w", err)
	}
	if status.DepartureGate == "" && status.CheckInDesks == "" {
		return "", nil
	}
	return updatedAt.Format(consts.ISO8601OutputTimeFormat), nil
}
