package trips

import (
	"sort"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/point"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	tripmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
)

type RestrictionsProvider struct {
	logger        log.Logger
	spanHelper    *span.Helper
	pointResolver *point.Resolver
	pointFactory  *point.Factory
}

func NewRestrictionsProvider(
	logger log.Logger,
	spanHelper *span.Helper,
	pointResolver *point.Resolver,
	pointFactory *point.Factory,
) *RestrictionsProvider {
	return &RestrictionsProvider{
		logger:        logger.WithName("api.trips.RestrictionsProvider"),
		spanHelper:    spanHelper,
		pointResolver: pointResolver,
		pointFactory:  pointFactory,
	}
}

func (p RestrictionsProvider) MakeBlock(trip *tripmodels.Trip) *apimodels.RestrictionsBlock {
	if trip == nil {
		return nil
	}

	spans := p.spanHelper.ReduceTransfers(trip.GetActiveSpans())
	visits := span.ExtractVisitsFromSpans(spans)
	sort.Stable(visitsByWhen(visits))

	russiaPoint, _ := p.pointFactory.MakeByGeoID(consts.RussiaGeoID)
	visits = p.wrapVisitsWithPoint(visits, russiaPoint)
	return p.findEdgeCrossingAndMakeBlock(visits)
}

func (p RestrictionsProvider) wrapVisitsWithPoint(visits []models.Visit, point models.Point) []models.Visit {
	leftTime := time.Now()
	rightTime := time.Now()

	if len(visits) > 0 {
		leftTime = visits[0].When()
		rightTime = visits[len(visits)-1].When()
	}

	leftVisit := models.NewVisit(point, leftTime.Add(-consts.Day))
	rightVisit := models.NewVisit(point, rightTime.Add(consts.Day))

	return append(
		append(
			[]models.Visit{leftVisit},
			visits...,
		),
		rightVisit,
	)
}

func (p RestrictionsProvider) findEdgeCrossingAndMakeBlock(visits []models.Visit) *apimodels.RestrictionsBlock {
	now := time.Now()

	isRussia := func(v models.Point) bool {
		return v.GetGeoID() == consts.RussiaGeoID
	}

	for i := 0; i < len(visits)-1; i++ {
		left := visits[i]
		right := visits[i+1]
		if right.When().Before(now) {
			continue
		}
		leftCountry, err := p.pointResolver.GetCountry(left.Point())
		if err != nil {
			p.logger.Error(
				"unable to resolve point to country",
				log.String("point_key", point.ExtractPointKey(left.Point())),
				log.Error(err),
			)
			continue
		}
		rightCountry, err := p.pointResolver.GetCountry(right.Point())
		if err != nil {
			continue
		}

		if isRussia(leftCountry) != isRussia(rightCountry) {
			return p.makeBlockFromCountries(leftCountry, rightCountry)
		}
	}
	return nil
}

func (p RestrictionsProvider) makeBlockFromCountries(fromCountry, toCountry models.Point) *apimodels.RestrictionsBlock {
	return &apimodels.RestrictionsBlock{
		FromCountryTitle: fromCountry.GetTitle(),
		FromGeoID:        fromCountry.GetGeoID(),
		ToCountryTitle:   toCountry.GetTitle(),
		ToGeoID:          toCountry.GetGeoID(),
	}
}

type visitsByWhen []models.Visit

func (v visitsByWhen) Len() int {
	return len(v)
}

func (v visitsByWhen) Less(i, j int) bool {
	return v[i].When().Before(v[j].When())
}

func (v visitsByWhen) Swap(i, j int) {
	v[i], v[j] = v[j], v[i]
}
