package trips

import (
	"context"
	"fmt"
	"net/url"
	"path"
	"sort"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	tripsapi "a.yandex-team.ru/travel/komod/trips/api/trips/v1"
	"a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/displaytime"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/consts"
	"a.yandex-team.ru/travel/komod/trips/internal/models"
	"a.yandex-team.ru/travel/komod/trips/internal/orders"
	"a.yandex-team.ru/travel/komod/trips/internal/references"
	"a.yandex-team.ru/travel/komod/trips/internal/span"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/containers"
	"a.yandex-team.ru/travel/library/go/strutil"
	"a.yandex-team.ru/travel/proto/dicts/rasp"
)

type TripsMapper struct {
	logger                  log.Logger
	raspMediaURL            string
	spansHelper             spansHelper
	pointImagesExtractor    pointImagesExtractor
	trainDescriptionBuilder *TrainDescriptionBuilder
	carriersRepo            *references.CarrierRepository
}

type spansHelper interface {
	ReduceTransfers([]models.Span) []models.Span
	ExtractVisitsRemovingExtremes([]models.Span) []models.Visit
	RemoveDuplicatedVisits([]models.Visit) []models.Visit
}

type pointImagesExtractor interface {
	Extract(point models.Point, alias consts.ImageAlias) string
}

func NewTripsMapper(
	logger log.Logger,
	spansHelper spansHelper,
	raspMediaURL string,
	pointImagesExtractor pointImagesExtractor,
	trainDescriptionBuilder *TrainDescriptionBuilder,
	carriersRepo *references.CarrierRepository,
) *TripsMapper {
	return &TripsMapper{
		logger:                  logger,
		carriersRepo:            carriersRepo,
		raspMediaURL:            raspMediaURL,
		spansHelper:             spansHelper,
		pointImagesExtractor:    pointImagesExtractor,
		trainDescriptionBuilder: trainDescriptionBuilder,
	}
}

func (m *TripsMapper) MapTrips(tripsList tripsmodels.Trips, userTime time.Time, alias consts.ImageAlias) []apimodels.TripItemRsp {
	result := make([]apimodels.TripItemRsp, len(tripsList))
	for i, trip := range tripsList {
		result[i] = m.mapTripItemRsp(trip, userTime, alias)
	}
	return result
}

func getDisplayDate(trip *tripsmodels.Trip, userTime time.Time) string {
	spans := trip.GetActiveOrAllSpans()
	return displaytime.NewBuilder().BuildDatesPair(
		displaytime.NewDatesPairArgs(
			userTime,
			span.GetStartTime(spans),
			span.GetEndTime(spans),
		).SetUseYear().SetUseRelative().SetShortMonth(),
	)
}

func (m *TripsMapper) mapTripItemRsp(trip *tripsmodels.Trip, userTime time.Time, alias consts.ImageAlias) apimodels.TripItemRsp {
	visits := m.extractActiveOrAllVisitsFromTrip(trip)
	image := m.extractTripImage(visits, alias)
	title := extractTripTitle(visits)
	displayDate := getDisplayDate(trip, userTime)

	if trip.ID == "" {
		return &apimodels.OrderTripItemRsp{
			OrderID:     trip.GetOrderIDs()[0].String(),
			Image:       image,
			Title:       title,
			DisplayDate: displayDate,
			State:       trip.State(),
		}
	} else {
		return &apimodels.RealTripItemRsp{
			ID:          trip.ID,
			Image:       image,
			Title:       title,
			DisplayDate: displayDate,
			OrderIDs:    extractOrderIDs(trip),
			State:       trip.State(),
		}
	}
}

func extractOrderIDs(trip *tripsmodels.Trip) []string {
	orderIDs := make([]string, 0, len(trip.OrderInfos))
	for _, o := range trip.OrderInfos {
		orderIDs = append(orderIDs, o.ID.String())
	}
	return orderIDs
}

func (m *TripsMapper) extractActiveOrAllVisitsFromTrip(trip *tripsmodels.Trip) []models.Visit {
	return m.spansHelper.RemoveDuplicatedVisits(
		m.spansHelper.ExtractVisitsRemovingExtremes(
			m.spansHelper.ReduceTransfers(
				trip.GetActiveOrAllSpans(),
			),
		),
	)
}

func (m *TripsMapper) extractTripImage(tripVisits []models.Visit, alias consts.ImageAlias) string {
	if len(tripVisits) == 0 {
		return ""
	}
	return m.pointImagesExtractor.Extract(tripVisits[0].Point(), alias)
}

func extractTripTitle(tripVisits []models.Visit) string {
	if len(tripVisits) == 0 {
		return ""
	}
	var titles []string
	for _, visit := range tripVisits {
		titles = append(titles, visit.Point().GetTitle())
	}
	return strings.Join(titles, " — ")
}

func (m *TripsMapper) MapBlocksRspByVertical(
	ctx context.Context,
	ordersList []orders.Order,
	userTime time.Time,
) []apimodels.BlockRsp {
	var aviaBlock apimodels.AviaOrders
	var hotelBlock apimodels.HotelOrders
	var trainBlock apimodels.TrainOrders
	var busBlock apimodels.BusOrders
	for _, order := range ordersList {
		switch model := order.(type) {
		case *orders.AviaOrder:
			aviaBlock.Values = append(aviaBlock.Values, m.mapAviaOrder(model, userTime))
		case *orders.TrainOrder:
			trainBlock.Values = append(trainBlock.Values, m.mapTrainOrder(ctx, model, userTime))
		case *orders.BusOrder:
			busBlock.Values = append(busBlock.Values, m.mapBusOrder(model, userTime))
		case *orders.HotelOrder:
			hotelBlock.Values = append(hotelBlock.Values, m.mapHotelOrder(model, userTime))
		}
	}
	return []apimodels.BlockRsp{
		sortAviaOrders(aviaBlock),
		sortHotelOrders(hotelBlock),
		sortTrainOrders(trainBlock),
		sortBusOrders(busBlock),
	}
}

func sortAviaOrders(block apimodels.AviaOrders) apimodels.BlockRsp {
	getState := func(i int) tripsapi.TripOrderState {
		return block.Values[i].TripOrderState
	}
	getStart := func(i int) time.Time {
		return block.Values[i].ForwardDeparture
	}
	sort.SliceStable(block.Values, createSortFunc(getState, getStart))
	return block
}

func sortHotelOrders(block apimodels.HotelOrders) apimodels.BlockRsp {
	getState := func(i int) tripsapi.TripOrderState {
		return block.Values[i].TripOrderState
	}
	getStart := func(i int) time.Time {
		return block.Values[i].CheckinDate
	}
	sort.SliceStable(block.Values, createSortFunc(getState, getStart))
	return block
}

func sortTrainOrders(block apimodels.TrainOrders) apimodels.BlockRsp {
	getState := func(i int) tripsapi.TripOrderState {
		return block.Values[i].TripOrderState
	}
	getStart := func(i int) time.Time {
		return block.Values[i].ForwardDeparture
	}
	sort.SliceStable(block.Values, createSortFunc(getState, getStart))
	return block
}

func sortBusOrders(block apimodels.BusOrders) apimodels.BlockRsp {
	getState := func(i int) tripsapi.TripOrderState {
		return block.Values[i].TripOrderState
	}
	getStart := func(i int) time.Time {
		return block.Values[i].ForwardDeparture
	}
	sort.SliceStable(block.Values, createSortFunc(getState, getStart))
	return block
}

func createSortFunc(getState func(int) tripsapi.TripOrderState, getStart func(int) time.Time) func(int, int) bool {
	return func(i int, j int) bool {
		iState := getState(i)
		jState := getState(j)
		if iState == jState {
			return getStart(i).Before(getStart(j))
		}
		return iState == tripsapi.TripOrderState_TRIP_ORDER_STATE_CONFIRMED
	}
}

func (m *TripsMapper) mapAviaOrder(
	order *orders.AviaOrder,
	userTime time.Time,
) apimodels.AviaOrder {
	var companies []apimodels.Company
	for _, airline := range order.Carriers {
		parsedRaspMediaURL, _ := url.Parse(m.raspMediaURL)
		parsedRaspMediaURL.Path = path.Join(parsedRaspMediaURL.Path, airline.SvgLogo)
		companies = append(
			companies, apimodels.Company{
				Title:   airline.Title,
				LogoURL: parsedRaspMediaURL.String(),
				Color:   airline.LogoBgColor,
			},
		)
	}

	return apimodels.AviaOrder{
		ID: order.ID().String(),
		Title: getTransportOrderTitle(
			order.FromSettlement,
			order.ToSettlement,
			order.FromStation,
			order.ToStation,
			false,
		),
		ForwardDeparture: order.ForwardDeparture,
		DisplayDateForward: displaytime.NewBuilder().BuildDateTime(
			displaytime.NewDateTimeArgs(userTime, &order.ForwardDeparture).SetUseRelative(false),
		),
		DisplayDateBackward: displaytime.NewBuilder().BuildDateTime(
			displaytime.NewDateTimeArgs(userTime, order.BackwardDeparture).SetUseRelative(false),
		),
		Pnr: order.PNR,
		RegistrationURL: getOptionalRegistrationURL(
			order.Carriers,
			userTime.UTC(),
			order.ForwardDeparture,
			order.BackwardDeparture,
			m.carriersRepo,
		),
		Companies:      companies,
		State:          order.State(),
		TripOrderState: order.TripsOrderState(),
	}
}

func getOptionalRegistrationURL(
	carriers []*rasp.TCarrier,
	now,
	forwardDeparture time.Time,
	backwardDeparture *time.Time,
	carriersRepo *references.CarrierRepository,
) string {
	if !needToShowRegistrationURL(now, forwardDeparture, backwardDeparture) {
		return ""
	}
	for _, c := range carriers {
		if url := buildRegistrationURL(carriersRepo, c); url != "" {
			return url
		}
	}
	return ""
}

func needToShowRegistrationURL(now time.Time, forwardDeparture time.Time, backwardDeparture *time.Time) bool {
	if backwardDeparture == nil {
		return !forwardDeparture.Before(now)
	}
	return !backwardDeparture.Before(now)
}

func (m *TripsMapper) mapTrainOrder(
	ctx context.Context,
	order *orders.TrainOrder,
	userTime time.Time,
) apimodels.TrainOrder {
	trains := make([]apimodels.Train, 0)
	for _, train := range order.Trains {
		trains = append(
			trains,
			apimodels.Train{
				Number:      train.TrainInfo.Number,
				Description: m.trainDescriptionBuilder.Build(train),
			},
		)
	}

	return apimodels.TrainOrder{
		ID: order.ID().String(),
		Title: getTransportOrderTitle(
			order.FromSettlement,
			order.ToSettlement,
			order.FromStation,
			order.ToStation,
			true,
		),
		ForwardDeparture: order.ForwardDeparture,
		DisplayDateForward: displaytime.NewBuilder().BuildDateTime(
			displaytime.NewDateTimeArgs(userTime, &order.ForwardDeparture).SetUseRelative(false),
		),
		DisplayDateBackward: displaytime.NewBuilder().BuildDateTime(
			displaytime.NewDateTimeArgs(userTime, order.BackwardDeparture).SetUseRelative(false),
		),
		PrintURL:                     order.PrintURL,
		Trains:                       trains,
		HasTransferWithStationChange: m.hasTransferWithStationChange(ctx, order.Trains),
		State:                        order.State(),
		TripOrderState:               order.TripsOrderState(),
		RefundedTicketsCount:         order.RefundedTicketsCount,
	}
}

func (m *TripsMapper) hasTransferWithStationChange(ctx context.Context, trains []*orders.Train) bool {
	for _, group := range containers.IterateByWindow(trains, 2) {
		prevTrain := group[0]
		curTrain := group[1]
		if curTrain.FromStation == nil || prevTrain.ToStation == nil {
			ctxlog.Error(ctx, m.logger, "not found station for train")
			continue
		}
		if curTrain.TrainInfo.Direction != prevTrain.TrainInfo.Direction {
			continue
		}
		if curTrain.FromStation.Id != prevTrain.ToStation.Id {
			return true
		}
	}
	return false
}

func (m *TripsMapper) mapBusOrder(order *orders.BusOrder, userTime time.Time) apimodels.BusOrder {
	return apimodels.BusOrder{
		ID:               order.ID().String(),
		Title:            order.Title,
		Description:      order.Description,
		ForwardDeparture: order.ForwardDeparture,
		DisplayDateForward: displaytime.NewBuilder().BuildDateTime(
			displaytime.NewDateTimeArgs(userTime, &order.ForwardDeparture).SetUseRelative(false),
		),
		DownloadBlankToken:   order.DownloadBlankToken,
		CarrierName:          order.CarrierName,
		State:                order.State(),
		TripOrderState:       order.TripsOrderState(),
		RefundedTicketsCount: order.RefundedTicketsCount,
	}
}

func (m *TripsMapper) mapHotelOrder(order *orders.HotelOrder, userTime time.Time) apimodels.HotelOrder {
	const imageSize = "S1"
	return apimodels.HotelOrder{
		ID:          order.ID().String(),
		Title:       order.Title,
		Stars:       order.Stars,
		CheckinDate: order.CheckinDate,
		DisplayDates: displaytime.NewBuilder().BuildDatesPair(
			displaytime.NewDatesPairArgs(userTime, order.CheckinDate, order.CheckoutDate).SetUseRelative(false),
		),
		Address:        order.Address,
		DocumentURL:    order.DocumentURL,
		Image:          fmt.Sprintf(order.ImageURLTemplate, imageSize),
		Coordinates:    order.Coordinates,
		State:          order.State(),
		TripOrderState: order.TripsOrderState(),
	}
}

func getTransportOrderTitle(
	fromSettlement, toSettlement *rasp.TSettlement,
	fromStation, toStation *rasp.TStation,
	forTrain bool,
) string {
	fromTitle := getTransportOrderTitlePart(fromSettlement, fromStation, forTrain)
	toTitle := getTransportOrderTitlePart(toSettlement, toStation, forTrain)
	return fmt.Sprintf("%s — %s", fromTitle, toTitle)
}

func getTransportOrderTitlePart(settlement *rasp.TSettlement, station *rasp.TStation, forTrain bool) string {
	stationTitle := ""
	settlementTitle := ""
	if settlement != nil {
		settlementTitle = strutil.Coalesce(
			settlement.GetTitle().GetRu().GetNominative(),
			settlement.GetTitleDefault(),
		)
	}
	if station != nil {
		stationTitle = strutil.Coalesce(
			station.GetTitle().GetRu(),
			station.GetTitleRuNominativeCase(),
			station.GetTitleDefault(),
		)
	}
	if forTrain {
		return strutil.Coalesce(stationTitle, settlementTitle)
	}
	return strutil.Coalesce(settlementTitle, stationTitle)
}
