package weather

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/ctxlog"
	apimodels "a.yandex-team.ru/travel/komod/trips/internal/components/api/trips/models"
	"a.yandex-team.ru/travel/komod/trips/internal/services/weather"
	tripsmodels "a.yandex-team.ru/travel/komod/trips/internal/trips/models"
	"a.yandex-team.ru/travel/library/go/containers"
)

type Provider struct {
	config           Config
	client           weather.Client
	logger           log.Logger
	requestExtractor *RequestExtractor
}

func NewProvider(config Config, client weather.Client, logger log.Logger, requestExtractor *RequestExtractor) *Provider {
	return &Provider{
		config:           config,
		client:           client,
		logger:           logger.WithName("WeatherProvider"),
		requestExtractor: requestExtractor,
	}
}

func (p *Provider) MakeBlock(ctx context.Context, trip *tripsmodels.Trip) apimodels.BlockRsp {
	if trip.EndTime().Before(time.Now()) {
		return nil
	}
	ctx, cancel := context.WithTimeout(ctx, p.config.MakeBlockTimeout)
	defer cancel()

	params := p.requestExtractor.Extract(trip)
	if params == nil {
		return nil
	}

	data := p.getData(ctx, params.geoID, params.startDate, params.days)
	data = p.filterShortForecastWithClimate(data)
	return p.mapWeather(data)
}

func (p *Provider) getData(ctx context.Context, geoID int, startDate time.Time, days int) *weather.GetByGeoIDResponse {
	data, err := p.client.GetByGeoID(ctx, geoID, startDate, days)
	logFields := []log.Field{
		log.Int("days", days),
		log.String("startDate", startDate.String()),
		log.Int("geoID", geoID),
	}
	if err != nil {
		ctxlog.Error(ctx, p.logger, "getting weather failed", append(logFields, log.Error(err))...)
		return nil
	}
	if data == nil {
		ctxlog.Info(ctx, p.logger, "weather", logFields...)
		return nil
	}
	return data
}

func (p *Provider) mapWeather(data *weather.GetByGeoIDResponse) apimodels.BlockRsp {
	if data == nil {
		return nil
	}
	var items []*apimodels.WeatherItem
	for _, item := range data.Data.Items {
		items = append(items, &apimodels.WeatherItem{
			ImageURL:    item.ImageURL,
			Title:       item.Title,
			Description: item.Description,
			URL:         item.URL,
			ItemType:    string(item.ItemType),
		})
	}
	return &apimodels.WeatherBlock{
		Forecast: items,
	}
}

func (p *Provider) filterShortForecastWithClimate(data *weather.GetByGeoIDResponse) *weather.GetByGeoIDResponse {
	if data == nil {
		return data
	}

	forecastBlocksCount := containers.CountIf(
		data.Data.Items,
		isBlockType(weather.ForecastItemType),
	)
	climateBlocksCount := containers.CountIf(
		data.Data.Items,
		isBlockType(weather.ClimateItemType),
	)

	if climateBlocksCount > 0 && forecastBlocksCount < p.config.MinForecastBlocksWithClimate {
		data.Data.Items = containers.FilterBy(
			data.Data.Items,
			isBlockType(weather.ClimateItemType),
		)
	}
	return data
}

func isBlockType(t weather.ItemType) func(item *weather.GetByGeoIDResponseItem) bool {
	return func(item *weather.GetByGeoIDResponseItem) bool {
		return item.ItemType == t
	}
}
